;Interpreter of version 3 Z-code
; game files, for the TRS80 Model 100.
;
; Copyright (C) 2015 Clinton Reddekop.
; You can redistribute and/or modify
; this program under the terms of the
; GNU General Public License as
; published by the Free Software
; Foundation, either version 3 of the
; License, or (at your option) any
; later version.
; This program is distributed in the
; hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even
; the implied warranty of
; MERCHANTABILITY or FITNESS FOR A
; PARTICULAR PURPOSE. See the GNU
; General Public License (available at
; <http://www.gnu.org/licenses/gpl-3.0.html>)
; for details.
;
;References: *** YOU NEED THESE ***
; [1] http://inform-fiction.org/zmachine/standards/z1point1/index.html
;  (The Z-Machine Standards Document,
;   Version 1.0  22nd June 1997
;   - by Graham Nelson)
; [2] http://mirrors.ibiblio.org/interactive-fiction/infocom/z-machine/zspec02/zmach06e.pdf
;  (The Z-machine, And How To Emulate
;   It - by Marnix Klooster)


;  TODO
;
; * clean up!
; * support character codes > 126 i.e.
;   cursor u/d/l/r
;   function keys f1 to f12
;   keypad 0 to 9
;   "extra characters"
; * review references, make sure this
;   program conforms to the standard
;   as best it can
; * ?support transcript out stream 2?
; * ?support cmd file in stream 1?


; ********** RAM CONTENTS *********** ;
;                                     ;
; (See http://www.club100.org/library/doc/ramabout.html) ;
; (See http://bitchin100.com/wiki/index.php?title=Model_100/102_RAM_Pointers) ;
;                                     ;
; ***** In the "free section": ****** ;
; (Once startup code has run.)        ;
;  ... (lower addresses) ...          ;
;  ...                                ;
;  ...end of BASIC arrays             ;
; frespc:                             ;
;  optional extra space               ;
;  stack [stksize]                    ;
; dynbase:                             ;
;  game image dynamic memory [dynsize] ;
;  9 more bytes from game image [9]   ;
; zstk:                               ;
;  z-machine stack space [zstksize]   ;
; ramfuncs:                           ;
;  ram functions [ramfuncssize]       ;
; strspc:                             ;
;  BASIC string space...              ;
;  ...                                ;
;  ... (higher addresses) ...         ;
;                                     ;
; *********************************** ;

; ********* Z-MACHINE STACK ********* ;
;                                     ;
; z-machine stack grows upward        ;
; z-machine stack has 16-bit words    ;
;  which are stored BIG-ENDIAN        ;
; z-machine stack frame format:       ;
;                                     ;
;  ...  (lower addresses)  ...        ;
;          ...                        ;
;       highest word of caller's      ;
;           evaluation stack          ;
;       msword of CALL INFO           ;
;       lsword of CALL INFO           ;
;       WORDS USED                    ;
;  zlp: caller's zlp                  ;
;       first local var               ;
;          ...                        ;
;       last local var                ;
;       lowest word of eval stack     ;
;          ...                        ;
;       highest word of eval stack    ;
;  zsp: first unused stack location   ;
;          ...                        ;
;  ...  (higher addresses)  ...       ;
;                                     ;
; WHERE:                              ;
;                                     ;
; - WORDS USED is valid in all stack  ;
;  frames except the top one - we set ;
;  it in the caller's stack frame     ;
;  when creating a new stack frame    ;
;  - in the top one it is kept = 0    ;
;                                     ;
; - WORDS USED is equal to (zsp-zlp)  ;
;   at the time of the call           ;
;                                     ;
; - 'CALL INFO' refers to a           ;
; 32-bit integer with the format:     ;
;  bits 31..24: var to get return val ;
;  bits 23..22: # of args supplied    ;
;  bit 21:      unused                ;
;  bits 20..17: #local vars           ;
;  bits 16..0:  caller's zpc          ;
;                                     ;
; - 'args supplied' (binary) is       ;
;  001 if 1 arg was supplied;         ;
;  011 if 2 args were supplied; OR    ;
;  111 if 3 args were supplied        ;
;                                     ;
; - WORDS USED and bits 23..17  of    ;
;  CALL INFO are used only for saving ;
;  and restoring  the stack - see     ;
;  "Quetzal" in [1]  and the savegame ;
;  and restoregame functions          ;
;                                     ;
; *********************************** ;


; ******* MAIN ROM FUNCTIONS ******** ;
lcd    equ 4b44h
lnkfil equ 02146h
srccom equ 02089h
kilcom equ 01fd9h
makhol equ 06b6dh

; **** REX SYSTEM BLOCK ADDRESSES *** ;
;namdir equ 04400h   ;FOR REX v4.8
namdir equ 06400h  ;? FOR REX v4.9 +

; ******* SYSTEM RAM LOCATIONS ****** ;
inpbuf equ 0f685h
inpbufend equ 0f783h
altlcd equ 0fcc0h
altlcdend equ 0fe00h
frespc equ 0fbb6h
strspc equ 0f678h
filnam equ 0fc93h 
bintab equ 0fbb0h
vartab equ 0fbb2h

; *********** CONSTANTS ************* ;
stksize equ 200 ;TODO analyze size
;NOTE zstksize is bounds-checked using
; a bit mask in some places - don't
; change zstksize without fixing that
zstksize equ (2*1024)
ramfuncssize equ (ramfuncsromcopyend - ramfuncsromcopystart)


; **** START OF ROM (ALL BLOCKS) **** ;
; Lowest 256 here are the lowest 256  ;
; bytes of _ALL_ REX blocks.          ;

; **** START OF INTERPRETER CODE **** ;

    .list
    .aseg
    org 0
    
reset:
    ;interrupts should be disabled
    ; before we get here, but just in
    ; case they aren't:
    di
    jmp setup
    ds 4
rstrt1:
    ret
    ds 7
rstrt2:
    ret
    ds 7
rstrt3:
    ret
    ds 7
rstrt4:
    ds 4
inttrp: ;"trap" is low battery
    di
    push h      ;;stk: ra,shl
    lxi h,inttrp
    jmp int2
intbcr: ;barcode reader
    di
    push h      ;;stk: ra,shl
    lxi h,intbcr
    jmp int2
intudr: ;uart data ready
    di
    push h      ;;stk: ra,shl
    lxi h,intudr
    jmp int2
inttp: ;timer pulse
    di
    jmp inttp2
;;;0040h OPT ROM needs 54, 43 here
;;; TODO does it?  Why?
    db 54,43
;0042h OPT ROM name (5+1 chars)
; (packaging program puts it here)
gamename:
    ds 5
blocknum:
    db 0
    db 0
;0049h this byte indicates how many
; REX blocks make up the game
; (packaging program puts it here)
numblocks:
    db 0
inttp2:
    push h      ;;stk: ra,shl
    lxi h,inttp
;
int2:           ;;stk: ra,shl
    push psw    ;; ra,shl,saf
    push h      ;; ra,shl,saf,intvec
    lxi h,ramint2
    xthl    ;; ra,shl,saf,ramint2
    push h  ;; ra,shl,saf,ramint2,intvec
    lda 0ff45h
    ani 0feh
    sta 0ff45h ;ready for STDROM switch
    jmp ramint


;verifyblock: helper for verify
;inpars: a = REX block number
;        de=start adx in block
;        bc=#bytes to add into checksum
;        hl = checksum
;outpar: hl = updated checksum
;        REX block rexblock[0] as OPTROM
;(This function needs to be in lowest
; 256 bytes so that every REX block
; will have a copy of it.)
verifyblock:
    ;select REX block a
    di
    call selrex
    ei
    push h
    mov h,b
    mov l,c
    ; checksum on stack
    ; hl = # bytes to process
    ; de = start adx in block
    ;
    ;first process l bytes, then 256*h
    ; bytes
    mov b,l ;l bytes
    inr h
    xthl ;hl=checksum, h saved on tos
    inr b
    dcr b
    jz cklp
cklp1:
    ;add next byte into checksum
    ldax d
    inx d
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    dcr b
    jnz cklp1
    ;... then 256*h bytes
cklp:
    xthl
    dcr h
    jz verbrt
    xthl
    ;b is 0 - do 256 bytes
cklp2:
    ;add next byte into checksum
    ldax d
    inx d
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    dcr b
    jnz cklp2
    jmp cklp
verbrt:
    lda rexblock
    di
    call selrex
    ei
    pop h
    ret

;simple copy routine
;copy bc bytes from *hl to *de
memcopy:
    mov a,m
    stax d
    inx h
    inx d
    dcx b
    mov a,b
    ora c
    jnz memcopy
    ret

;setup routine - from reset
; get here using BASIC CALL 63012
;interrupts are disabled
setup:
    ;copy ram interrupt code
    lxi b,(rmicde-rmicd)
    lxi h,rmicd
    lxi d,ramint
    call memcopy
    ;now we can handle interrupts
    ei
    ;we must be in the first REX block
    ; to continue - it has last byte
    ; of name set to '0'
    lda blocknum
    cpi '0'
    jnz exit2
    jmp setup2


; ************** NOTE *************** ;
; Everything before this point MUST   ;
; fit in lowest 256 bytes, which are  ;
; put into all REX blocks.            ;
; Remainder of code in this file is   ;
; only in the first REX block, except ;
; as noted otherwise.                 ;

    org 256

waits: db 'One moment please...'
waitsend:

setup2:
    ;jump here to restart game
restart:
;;    ;temporary stack for startup
    lxi sp,tmpstackend
    ;get size of game image dynamic mem
    ;note big-endian
    lxi h,(roimg+0eh);statbase in hdr
    mov d,m
    inx h
    mov e,m
    xchg
    shld dynsize
    ;dynsize must be at least 64 bytes,
    ;at most 20480 bytes
    ;the packaging program should check
    ; this, but just in case...
    lxi d,(64-1)
    call cmp16
    jnc exit ; can't call error here
    lxi d,(20480+1)
    xchg
    call cmp16
    jnc exit ; can't call error here
    xchg
    call negate
    shld mdynsize ;mdynsize = -dynsize
    ;set up ram pointers
    lhld strspc
    xchg
    lxi h,ramfuncssize
    call sub16
    shld ramfuncs
    xchg
    lxi h,zstksize
    call sub16
    shld zstk
    xchg
    lxi h,9
    call sub16
    shld dyntop
    xchg
    lhld dynsize
    call sub16
    shld dynbase
    ;make sure we have enough memory
    ; we need
    ; (dynbase - stksize) >= frespc
    xchg
    lxi h,stksize
    call sub16
    xchg
    lhld frespc
    call cmp16
    jc exit ; can't call error here
    ;stack in [frespc,(dynbase-1))
    lhld dynbase
    dcx h ;stack grows down from here
    sphl
    ;copy ram functions into ram
    lhld ramfuncs ;dst
    xchg
    lxi h,ramfuncsromcopystart ;src
    lxi b,ramfuncssize
    call memcopy
    ;
    ;set up jump table for ram funcs
    lxi b,rfjmpt ;dst
    lxi h,offsettablestart ;src
jumptablesetuploop:
    ;write jmp opcode
    mvi a,0c3h
    stax b
    inx b
    ;read offset relative to start of
    ; ram functions into de
    mov e,m
    inx h
    mov d,m
    inx h
    ;add offset to start address of ram
    ; functions, write as target of jmp
    push h
    lhld ramfuncs
    dad d
    mov a,l
    stax b
    inx b
    mov a,h
    stax b
    inx b
    pop h
    ;check if table complete
    lxi d,offsettableend
    call cmp16
    jnz jumptablesetuploop
;;    ;
;;    ;use local stack
;;    lhld dynbase
;;    dcx h ;stack grows down from here
;;    sphl
    ;(from here on it is safe to call
    ; error)
    ;init for screen output - in case
    ; error is called
    call scinit
    ;print msg to wait
    mvi a,(waitsend-waits)
    lxi h,waits
    call printstring
    ;zero ram vars in ALTLCD
    lxi d,(zeroinitmemend-zeroinitmem)
    lxi h,zeroinitmem
    mvi b,0
dozero:
    mov m,b
    inx h
    dcx d
    mov a,d
    ora e
    jnz dozero
    mvi a,'.'
    call safelcd
    ; set up for accessing z-mach. mem
    call setupimage
    mvi a,'.'
    call safelcd
    ; check version
    ;  only version 3 supported
    ;the packaging program should check
    ; this, but just in case...
    lxi h,0
    call zgtblw
    cpi 3
    cnz error
    ; verify file checksum if any
    call verify
    mvi a,'.'
    call safelcd
    
    ; read z-machine address
    ;  variables from file header
    
    ; zpc  program counter
    lxi h,6h
    call zgtwlw
    mvi c,0
    xchg
    lxi h,zpc
    call putl
    
    ; zstatbase  static mem base
    lxi h,0eh
    call zgtwlw
    shld zstatbase
    
    ; zdictbase  dictionary entries base
    ; zdseps  dict. "word-separators"
    ; zdentrylength  dict. entry length
    lxi h,8h
    call zgtwlw
    call zgtblw
    inx h
    mov b,a ;# "word-separators"
    cpi (zdsepsend-zdseps)
    cnc error
    lxi d,zdseps
    ana a
    jz zdnocopy
    ; copy "word-separators"
zdsepscopy:
    call zgtblw
    inx h
    stax d
    inx d
    dcr b
    jnz zdsepscopy
zdnocopy:
    xra a
    stax d
    ; copy entry length
    call zgtblw
    inx h
    sta zdentrylength
    ; copy # entries
    push h
    call zgtwlw
    shld zdnumentries
    pop h
    inx h ;now hl is base of entries
    inx h
    shld zdictbase
    
    ; zobjectsbase  object base
    lxi h,0ah
    call zgtwlw
    shld zobjectsbase
    
    ; zobjectstable  object tree
    ;  zobjectstable = zobjectsbase + 31*2 - 9
    ;  (9 byte positions before obj 1,
    ;   since there is no object 0)
    ;  must be in lowest 64k so we
    ;   know this won't overflow
    lxi d,(31*2-9)
    dad d
    shld zobjectstable
    
    ; zglobalsbase  globals base
    lxi h,0ch
    call zgtwlw
    shld zglobalsbase
    
    ; zabbrevsbase  abbreviations base
    lxi h,18h
    call zgtwlw
    shld zabbrevsbase
    
    ;set up flags & std rev # in header
    call hdrstp
    jmp stpcont
    
hdrstp:
    ;set flags in header indicating
    ; (non-)availability of features
    ; flags 1:
    lxi h,1
    call zgtblw
    ori 10h ; no status line
    ani 9fh ; no screen split,
            ; no var-width font
    call zptblw
    ; flags2:
    lxi h,10h
    call zgtblw
    ani 0feh ; transcript off
    ori 02h  ; fixed-width font on
    call zptblw
    
    ;write standard revision number (1.0)
    lxi h,32h
    mvi a,01h
    call zptblw
    lxi h,33h
    mvi a,00h
    call zptblw
    
    ret
    
stpcont:
    
    ;init zsp, zlp
    lxi h,0
    shld zsp
    shld zlp
    
    ; init for screen output
    call scinit
    ; start of [more] block
    call morebegin
    ; put random number generator in
    ;  "unpredictable" mode
    lxi h,0
    call randgen
    
    ; create a dummy z-machine stack
    ;  frame to be the current frame
    ;  when we start running
    ; this is needed for saving/rest'g
    ;  the z-machine stack
    lxi h,3
    shld zlp
    lxi h,0
    call zpush ; CALL INFO MSword
    call zpush ; CALL INFO LSword
    call zpush ; WORDS USED
    call zpush ; caller's zlp
    
;TODO REMOVE THIS TEST CODE
; TEST CODE
; Check that stack ptr always the same
; at start of runloop.  We do this by
; expecting a specific value at *sp.
    lxi d, 0a535h
    push d
; END TEST CODE

;loop, executing z-machine operations
runloop:

;TODO REMOVE THIS TEST CODE
; TEST CODE
; Check that stack ptr always the same
; at start of runloop.  We do this by
; expecting a specific value at *sp.
    lxi d, 0a535h
    pop h
    push h
    call sub16
    mov a,h
    ora l
    cnz error
; END TEST CODE

    ;default opcount to 0
    xra a
    sta opcount
    
;;    ;read opcode from *zpc and inc zpc
;;    call rbizpc
    ;read 10bytes from *zpc into instbuf
    ;10 bytes is big enough to cover
    ; from opcode to last operand in
    ; all cases
    call readinst
    ;opcode byte
    call getinstbufbyte

    sta opcode
    mov b,a ;opcode saved in b
    
    ;check form of instruction
    ani 0c0h
    cpi 080h
    jz short
    cpi 0c0h
    jnz long
    
variable: ; VARIABLE FORM instruction
    ;get opnum
    mov a,b ;opcode
    ani 1fh
    sta opnum
;;    ;get operand types byte from *zpc
;;    ; and inc zpc
;;    call rbizpc
    ;operand types byte
    call getinstbufbyte
    ;get operands
    ;a is operand types byte
    mvi b,4
    lxi d,operands
varoploop:
    rlc
    rlc
    mov c,a
    ani 03h ;type of operand
    call getoperand ;value in hl
    mov a,c
    xchg ;val in de, opnd is *hl
    mov m,e
    inx h
    mov m,d
    inx h
    xchg ;next opnd is *de 
    dcr b
    jnz varoploop
    
    ;set count
    lda opcode
    mov b,a
    ani 20h
    jz vcnt2
    ;VAR instruction
    mvi a,3
    sta count
    jmp endform
vcnt2:
    ;2OP instruction
    mvi a,2
    sta count
    jmp endform

short: ; SHORT FORM instruction
    ;get opnum
    mov a,b ;opcode
    ani 0fh
    sta opnum
    ;get operand if any
    mov a,b
    rrc
    rrc
    rrc
    rrc
    ani 03h
    call getoperand
    shld op1
    ;set count = opcount
    lda opcount
    sta count
    jmp endform

long: ; LONG FORM instruction
    ;get opnum
    mov a,b ;opcode
    ani 1fh
    sta opnum
    ;get 1st operand
    mvi a,40h
    ana b
    jz smcst1
    ;1st operand is a variable
    call getvar
    jmp lop2
smcst1:
    ;1st operand is a small constant
    call getsmallconst
lop2:
    shld op1
    ;get 2nd operand
    mvi a,20h
    ana b
    jz smcst2
    ;2nd operand is a variable
    call getvar
    jmp endlop
smcst2:
    ;2nd operand is a small constant
    call getsmallconst
endlop:
    shld op2
    ;set count and opcount to 2
    mvi a,2
    sta count
    sta opcount
    
endform:
    ;increment zpc past all the bytes
    ; we used
    call fixzpc
    ; now count and opnum determine
    ;  the operation to perform
    lda opnum
    mov c,a
    mvi b,0 ;opnum in bc
    lda count
    ana a
    jz is0op
    dcr a
    jz is1op
    dcr a
    jz is2op

isvar:
    ; VAR instruction
    mov a,c
    cpi 16h
    cnc error ;error if opnum > 15h
    lxi h,vartable
    ;jmp to vartable[opnum]
setpc:
    dad b
    dad b ; hl = table + 2*opnum
    mov e,m
    inx h
    mov d,m
    lxi h,runloop
    push h ; will be used to return
           ; to runloop
    push d
    ret ; jmp to address from table
    
vartable:
    dw var00, var01, var02, var03
    dw var04, var05, var06, var07
    dw var08, var09, ignor, ignor
    dw errop, errop, errop, errop
    dw errop, errop, errop, ignor
    dw ignor, ignor
    
is2op:
    ;2OP instruction
    mov a,c
    cpi 19h
    cnc error ;error if opnum > 18h
    lxi h,twotable
    jmp setpc ; jmp to twotable[opnum]
    
twotable:
    dw errop, two01, two02, two03
    dw two04, two05, two06, two07
    dw two08, two09, two0a, two0b
    dw two0c, two0d, two0e, two0f
    dw two10, two11, two12, two13
    dw two14, two15, two16, two17
    dw two18
    
is1op:
    ;1OP instruction
    mov a,c
    cpi 10h
    cnc error ;error if opnum > 0fh
    lxi h,onetable
    jmp setpc ; jmp to onetable[opnum]
    
onetable:
    dw one00, one01, one02, one03
    dw one04, one05, one06, one07
    dw errop, one09, one0a, one0b
    dw one0c, one0d, one0e, one0f
    
is0op:
    ;0OP instruction
    mov a,c
    cpi 0eh
    cnc error ;error if opnum > 0dh
    lxi h,zrotable
    jmp setpc ; jmp to zrotable[opnum]
    
zrotable:
    dw zro00, zro01, zro02, zro03
    dw zro04, zro05, zro06, zro07
    dw zro08, zro09, zro0a, zro0b
    dw ignor, zro0d
    

errop:
    pop h ;since we left runloop on stk
    call error
    
ignor:  ;jmp to runloop
    ret
    

var00:
  ;call routine...up to 3 args...
  ; -> result
    lhld op1
    mov a,h
    ora l
    jnz setuproutine
    lxi h,0
    jmp putvar
    
var01:
  ;storew array word-index value
  ;stores always in lower 64k of zmem
    lhld op2
    xchg    ; de = word-index
    lhld op1; hl = array
    dad d
    dad d   ; 0hl = &array[word-index]
    xchg
    lhld op3
    xchg
    jmp zptwlw
    
var02:
  ;storeb array byte-index value
  ;stores always in lower 64k of zmem
    lhld op1
    xchg    ; de = array
    lhld op2
    dad d   ; 0hl = &array[byte-index]
    lda op3 ; LSbyte of op3
    jmp zptblw
    
var03:
  ;put_prop object property value
    lda op2
    mov b,a
    lda op1
    call getpropadx
    ani 0e0h ; prop len bits
    jnz v3bigp
    ; property length is 1
    lda op3
    jmp zptblw
v3bigp:
    ; property length > 1
    xchg
    lhld op3
    xchg
    jmp zptwlw
    
var04:
  ;sread text parse
    lhld op2 ; parse buf
    xchg
    lhld op1 ; text buf
    jmp rdline
    
var05:
  ;print_char output-character-code
    lda op1 ; LSbyte of op1
    jmp pzscii
    
var06:
  ;print_num value
    jmp printnum
    
var07:
  ;random range -> (result)
    lhld op1
    call randgen
    jmp putvar
    
var08:
  ;push value
    lhld op1
    jmp zpush
    
var09:
  ;pull (variable)
    call zpop
    lda op1 ; LSbyte of op1
    jmp putgivenvar

    
two01:
  ;je a b ?(label)
    lda opcount
    sui 2
    rc ; opcount < 2 -- never jump
    lhld op1
    xchg ; op1 in de
    jz t1eq2 ; opcount == 2
    dcr a
    jz t1eq3 ; opcount == 3
    ;opcount == 4
    ; branch(true) if
    ;   op1==op4 || op1==op3 || op1==op2
    lhld op4
    call cmp16s
    jnz t1eq3
    call branchtrue
    jmp t1cret
t1eq3:
    ; branch(true) if
    ;   op1==op3 || op1==op2
    lhld op3
    call cmp16s
    jnz t1eq2
    call branchtrue
    jmp t1cret
t1eq2:
    ; branch(true) if
    ;   op1==op2
    lhld op2
    call cmp16s
    jnz t1ne
    call branchtrue
    jmp t1cret
t1ne:
    call branchfalse ; branch(false)
    ;
    ; check if branch told us to return
    ;  from the current z-routine
t1cret:
    ;; if a {
    ;;          if zlp == 0 {
    ;;                  end game
    ;;          }
    ;;          TeardownRoutine()
    ;;          (save return value)
    ;; }
    ana a
    rz ; no return from curr. z-routine
t1ret:
    push h ; save ret val from branch
    lhld zlp
    mov a,h
    ora l
    jz t1end ; not in a z-routine
    ; return from curr. z-routine
    call teardownroutine
    ;a is now dest var # for res
    pop h ; val to return, from branch
    jmp putgivenvar
    
t1end: ; end the game
    pop h ; ret val from branch
    pop h ; since runloop left on stk
    jmp endgame
    
two02:
  ;jl a b ?(label)
    lhld op1
    xchg
    lhld op2 ; op1 in de, op2 in hl
    call cmp16s ; op1 - op2
    cc branchtrue  ; branch(true)
    cnc branchfalse ; branch(false)
    jmp t1cret
    
two03:
  ;jg a b ?(label)
    lhld op2
    xchg
    lhld op1 ; op2 in de, op1 in hl
    call cmp16s ; op2 - op1
    cc branchtrue  ; branch(true)
    cnc branchfalse ; branch(false)
    jmp t1cret
    
two04:
  ;dec_chk (variable) value ?(label)
    lda op1 ; LSbyte of op1
    call getgivenvar
    dcx h
    push h ; save decremented var
    lda op1 ; LSbyte of op1
    call putgivenvar
    pop h ; var
    xchg
    lhld op2
    call cmp16s ; var - op2
    cc branchtrue
    cnc branchfalse
    jmp t1cret
    
two05:
  ;inc_chk (variable) value ?(label)
    lda op1 ; LSbyte of op1
    call getgivenvar
    inx h
    push h ; save incremented var
    lda op1 ; LSbyte of op1
    call putgivenvar
    lhld op2
    xchg
    pop h ; var
    call cmp16s ; op2 - var
    cc branchtrue
    cnc branchfalse
    jmp t1cret
    
two06:
  ;jin obj1 obj2 ?(label)
    lda op1
    call goepar
    mov b,a
    lda op2
    cmp b
    cz branchtrue
    cnz branchfalse
    jmp t1cret
    
two07:
  ;test bitmap flags ?(label)
    lhld op1
    xchg
    lhld op2
    mov a,d
    ana h
    xra h
    mov d,a
    mov a,e
    ana l
    xra l
    ora d
    cz branchtrue
    cnz branchfalse
    jmp t1cret
    
two08:
  ;or a b -> (result)
    lhld op1
    xchg
    lhld op2
    mov a,d
    ora h
    mov h,a
    mov a,e
    ora l
    mov l,a
    jmp putvar
    
two09:
  ;and a b -> (result)
    lhld op1
    xchg
    lhld op2
    mov a,d
    ana h
    mov h,a
    mov a,e
    ana l
    mov l,a
    jmp putvar
    
two0a:
  ;test_attr object attribute ?(label)
    lda op1
    mov h,a
    call goeaf
    lda op2
    call testbit
    cnz branchtrue
    cz branchfalse
    jmp t1cret
    
two0b:
  ;set_attr object attribute
    lda op1
    mov h,a
    call goeaf
    lda op2
    call setbit
    lda op1
    mov h,a
    jmp poeaf
    
two0c:
  ;clear_attr object attribute
    lda op1
    mov h,a
    call goeaf
    lda op2
    call clearbit
    lda op1
    mov h,a
    jmp poeaf
    
two0d:
  ;store (variable) value
    lhld op2
    lda op1 ; LSbyte of op1
    jmp putgivenvar
    
two0e:
  ;insert_obj object destination
    jmp insobj
    
two0f:
  ;loadw array word-index -> (result)
    mvi c,0
    lhld op1
    xchg    ; cde = array
    lhld op2
    call ahl24
    call ahl24;cde=&array[word-index]
    call zgetw
    jmp putvar
    
two10:
  ;loadb array byte-index -> (result)
    mvi c,0
    lhld op1
    xchg    ; cde = array
    lhld op2
    call ahl24; cde=&array[byte-index]
    call zgetb ; a = array[byte-index]
    mov l,a
    mvi h,0
    jmp putvar
    
two11:
  ;get_prop object property -> (result)
    lda op2
    mov b,a
    lda op1
    call getpropadx
    ana a
    jz t11def ; not found
    ani 0e0h ; prop len bits
    jnz t11bgp ; prop length > 1
    ;prop length = 1
    call zgtblw
    mov l,a
    mvi h,0
    jmp putvar
t11def:
    ;prop not found - take default
    lhld zobjectsbase
    dcx h
    dcx h
    xchg
    lhld op2
    dad h
    dad d ;zobjectsbase - 2 + 2*op2
t11bgp:
    ; prop length > 1
    call zgtwlw
    jmp putvar
    
two12:
  ;get_prop_addr object property
  ; -> (result)
    lda op2
    mov b,a
    lda op1
    call getpropadx
    jmp putvar
    
two13:
  ;get_next_prop object property
  ; -> (result)
    lhld op2
    mov a,h
    ora l
    jz t13fst ; prop 0: get first prop
    ; get next property
    lda op2
    mov b,a
    lda op1
    call getpropadx
    ; skip to size byte of next prop
    rlc
    rlc
    rlc
    ani 7h ;length bits
    inr a ;actual length
    call aahl
t13pnm:
    call zgtblw ;get size byte
    ani 1fh ; mask prop number
    mov l,a
    mvi h,0
    jmp putvar
t13fst: ; get first property
    lda op1
    call gopta
    push h
    call zgtblw ; txt len byte
    mov l,a ; skip short name
    mvi h,0
    dad h   ; 2*(txt len)
    inx h   ;  + 1
    pop d
    dad d   ;  + adx of prop table
    jmp t13pnm
    
two14:
  ;add a b -> (result)
    lhld op1
    xchg
    lhld op2
    dad d
    jmp putvar
    
two15:
  ;sub a b -> (result)
    lhld op1
    xchg
    lhld op2
    call sub16
    jmp putvar
    
two16:
  ;mul a b -> (result)
    lhld op1
    xchg
    lhld op2
    call mul16s
    jmp putvar
    
two17:
  ;div a b -> (result)
    lhld op1
    xchg
    lhld op2
    call div16s ; result in de
    xchg ; result in hl
    jmp putvar
    
two18:
  ;mod a b -> (result)
    lhld op1
    xchg
    lhld op2
    call div16s ; result in hl
    jmp putvar
    

one00:
  ;jz a ?(label)
    lhld op1
    mov a,h
    ora l
    cz branchtrue
    cnz branchfalse
    jmp t1cret
    
one01:
  ;get_sibling object
  ;            -> (result) ?(label)
    lda op1
    call goesib
one1ch:
    ana a
    push psw ;z flag
    mov l,a
    mvi h,0
    call putvar
    pop psw
    cnz branchtrue
    cz branchfalse
    jmp t1cret
    
one02:
  ;get_child object
  ;            -> (result) ?(label)
    lda op1
    call goefch
    jmp one1ch
    
one03:
  ;get_parent object -> (result)
    lda op1
    call goepar
    ana a
    mov l,a
    mvi h,0
    jmp putvar
    
one04:
  ;get_prop_len property-address
  ; -> (result)
    lhld op1
    dcx h
    call zgtblw; size byte
    rlc
    rlc
    rlc
    ani 7h
    inr a ;actual property length
    mov l,a
    mvi h,0
    jmp putvar
    
one05:
  ;inc (variable)
    lda op1
    call getgivenvar
    inx h
    lda op1
    jmp putgivenvar
    
one06:
  ;dec (variable)
    lda op1
    call getgivenvar
    dcx h
    lda op1
    jmp putgivenvar
    
one07:
  ;print_addr byte-address-of-string
    mvi c,0
    lhld op1
    xchg ; string address in cde
    jmp pzstr
    
one09:
  ;remove_obj object
    jmp remobj
    
one0a:
  ;print_obj object
    lda op1
    call gopta ;text-length
    inx h
    xchg
    mvi c,0 ;adx of short name in cde
    jmp pzstr
    
one0b:
  ;ret value
    lhld op1
    jmp t1ret
    
one0c:
  ;jump ?(label)
    lhld op1
    dcx h
    dcx h
    push h ; save (offset - 2)
    lxi h,zpc
    call getl ; zpc in cde
    pop h     ; (offset - 2) in hl
    call ashl24
    lxi h,zpc
    jmp putl

one0d:
  ;print_paddr packed-address-of-string
    mvi c,0
    lhld op1 ; hl = op1
    mov d,h
    mov e,l ; cde = op1
    call ahl24 ; cde = 2*op1
    jmp pzstr
    
one0e:
  ;load (variable) -> (result)
    lda op1 ; LSbyte of op1
    call getgivenvar
    jmp putvar
    
one0f:
  ;not value -> (result)
    lhld op1
    mov a,h
    cma
    mov h,a
    mov a,l
    cma
    mov l,a
    jmp putvar
    

zro00:
  ;rtrue
    lxi h,1
    jmp t1ret

zro01:
  ;rfalse
    lxi h,0
    jmp t1ret

zro02:
  ;print
    lxi h,zpc
    call getl
    call pzstr
    lxi h,zpc
    jmp putl
    
zro03:
  ;print_ret
    lxi h,zpc
    call getl
    call pzstr
    lxi h,zpc
    call putl
    mvi a,13
    call pzscii
    lxi h,1
    jmp t1ret
    
zro04:
  ;nop
    ret
    
zro05:
  ;save ?(label)
    call savegame
    cc branchtrue  ; branch(true)
    cnc branchfalse ; branch(false)
    jmp t1cret
    
zro06:
  ;restore ?(label)
    call restoregame
    cc branchtrue  ; branch(true)
    cnc branchfalse ; branch(false)
    jmp t1cret
    
zro07:
  ;restart
    jmp restart
    
zro08:
  ;ret_popped
    call zpop
    jmp t1ret
    
zro09:
  ;pop
    jmp zpop ; just does zsp -= 1
    
zro0a:
  ;quit
    pop h ; since runloop left on stk
    jmp endgame
    
zro0b:
  ;new_line
    mvi a,13
    jmp pzscii
    
zro0d:
  ;verify ?(label)
    ; we do the verify when the program
    ;  is starting, so don't need to
    ;  do it here - just do the branch
    call branchtrue
    jmp t1cret
    
;handles errors
;NOTE this function may NOT be called
; early during setup, before ram vars,
; ram subroutines etc. are set up
;DOESN'T RETURN
error:
    lxi h,errst
errlp:
    mov a,m
    inx h
    ana a
    jz derrst
    call safelcd
    jmp errlp
derrst:
    ;print address of caller
    pop h
    dcx h
    dcx h
    dcx h ;adx error was called from
    mov a,h
    call prthex
    mov a,l
    call prthex
    ;advance line
    mvi a,13
    call safelcd
    mvi a,10
    call safelcd
    call safechget
    jmp exit
errst: db 13,10,'  ERROR #',0
hexdig: db '0123456789abcdef'
;helper for error
;DESTROYS de
prthex:
    push h
    push psw
    rrc
    rrc
    rrc
    rrc
    ani 0fh
    lxi d,hexdig
    add e
    mov e,a
    mvi a,0
    adc d
    mov d,a
    ldax d
    call safelcd
    pop psw
    ani 0fh
    lxi d,hexdig
    add e
    mov e,a
    mvi a,0
    adc d
    mov d,a
    ldax d
    call safelcd
    pop h
    ret
    
;handles end of game & program
endgame:
    ;do cleanup
    call clnimg ;game image
    call moreend ;flush buffered text
    call sccln  ;screen
    jmp exit


; z-machine alphabets ('?' for values
; that should never be taken from here)
; NOTE newline is 13 in ZSCII
za0:
  db '??????abcdefghijklmnopqrstuvwxyz'
za1:
  db '??????ABCDEFGHIJKLMNOPQRSTUVWXYZ'
za2:
  db '???????',13,'0123456789.,!?_#'
  db 27h, 22h, '/', 5ch, '-:()'


;random number generator
;inpar: hl
;outpar: hl
;we will call the inpar "ip" and the
; outpar "op" below to distinguish
; them
;  ip=0: restart "unpredictable" mode
;           op=0
;  ip>0: get a random int in [1, ip]
;           op is this random int
;  ip<0: restart "predictable" mode
;         in which the generator
;         internally generates
;         1, 2, ...abs(ip), 1, ...
;           op=0
;
;Reference: we use this LCG
; Xn+1 = 214013*Xn + 2531011
;(Xn+1 = 343fdh*Xn + 269ec3h)
; as given here:
;http://rosettacode.org/wiki/Linear_congruential_generator
;
;MODIFIES ALL REGS
randgen:
    ; check sign of inpar
    mov a,h
    ana a
    jm rstpd
    ora l
    jz rstupd
    ;
    ; inpar is positive
    xchg ; inpar in de
    ; check mode
    lhld predseed
    mov a,h
    ora l
    jz updmod
    ;
    ; "predictable" mode
    push d ; inpar on stack
    lhld predseed
    dcx h
    xchg
    lhld prevrand
    inx h
    call cmp16 ; de - hl
    jnc prndok
    lxi h,0
prndok: ;hl is in [0,predseed-1]
    shld prevrand
    jmp outrnd
    ;
updmod:
    ; "unpredictable" mode
    push d ;inpar
    ; multiply the prev gen'd value
    ;  by 343fdh
    lhld prevrand     ; low 16 bits
    xchg
    lxi b,43fdh     ; low 16 bits
    call mul16u
    shld randtemp     ; low 16 of result
    xchg
    shld (randtemp+2) ; high 16 of result
    lhld (prevrand+2) ; high 16 bits
    xchg
    call mul16u
    ; add low 16 bits of res into
    ;  high 16 bits of randtemp
    xchg
    lhld (randtemp+2) ; high 16 bits
    dad d
    shld (randtemp+2)
    ; now randtemp = prevrand * 43fdh
    ; add prevrand * 30000h,
    ; truncating result to 32 bits
    lhld prevrand     ; low 16 bits
    mov d,h
    mov e,l
    dad d
    dad d ; hl=3*(low 16 bits prevrand)
    xchg
    lhld (randtemp+2) ; high 16 bits
    dad d
    shld (randtemp+2) ; high 16 bits
    ; done multiply - now add 269ec3h
    ; and put result in prevrand,
    ; truncating result to 32 bits
    lxi d,9ec3h     ; low 16 bits
    lhld randtemp     ; low 16 bits
    dad d ; might have cy
    shld prevrand     ; low 16 bits
    lxi d,0026h     ; high 16 bits
    lhld (randtemp+2) ; high 16 bits
    jnc updnoc
    inx h ; carry from low 16 bits add
updnoc:
    dad d
    shld (prevrand+2) ; high 16 bits
    ; the new random number has now
    ; been generated and put in
    ; prevrand
    ; use 15 LSbits of the 16 MSbits
    ; of prevrand to get the return val
    ; 16 MSbits of prevrand are still
    ; in hl
    mov a,h
    ani 7fh
    mov h,a
    jmp outrnd
    ;
rstpd:
    ; restart "predictable" mode
    ; inpar is in hl
    call negate
    shld predseed  ; seed
    lxi h,0ffffh ; inc's to 0 next call
    shld prevrand
    lxi h,0
    ret
    ;
rstupd:
    ; restart "unpredictable" mode
    ; hl is still 0
    shld predseed ;0 indicates unpred.
    ; seed the generator
    lda 0f92fh ; timer, 0..125
    lxi h,prevrand
    ;I just made up the values xored...
    mov m,a  ;timer
    inx h
    xri 0aah
    mov m,a  ;timer xor aah
    inx h
    xri 0a5h
    mov m,a  ;timer xor 0fh
    inx h
    xri 3ch
    mov m,a  ;timer xor 33h
    ;return 0
    lxi h,0
    ret
outrnd:
    pop d ;inpar
    ; hl has internally gen'd value
    ; return (hl mod de) + 1
    xchg
    call div16u
    inx h
    ret
    
fnprompt:   db 'enter filename: '
fnpromptend:

;saves the game
;inpars: none
;outpars:
; cy=1 if successful
; cy=0 if not
;MODIFIES ALL REGS
savegame:
    ;FIRST PASS: dosave is run with a
    ; version of svwrite which doesn't
    ; actually write to file; instead
    ; it keeps track of size of
    ; file that dosave tries to write
    
    ;set up for svwrite, svwriteto
    lxi h,0
    shld svwnext
    shld svwritefilestart
    mvi a,0c3h ;'jmp' opcode
    sta svwrite
    lxi h,svwrite1
    shld (svwrite+1)
    
    ;set up for svreaddyn, svreaddynend
    lhld dynbase
    shld svrdnext
    
    ;set up for readgam
    lxi h,2f00h ;start of game image
    shld rgnext
    
    ;do first pass
    call dosave
    ;now svwnext has needed file size
    
    ;determine if we have enough space
    ; for the save file
    ;we need
    ; (dynbase - stksize) >=
    ;              frespc + svwnext + 6
    ; (+ 6 is for CO file header)
    lhld dynbase
    xchg
    lxi h,stksize
    call sub16
    push h
    lhld frespc
    xchg
    lhld svwnext
    dad d
    jc sgfail
    pop d
    call cmp16
    jc sgfail
    
    ;get filename
    mvi a,(fnpromptend-fnprompt)
    lxi h,fnprompt
    call printstring
    call getfilename
    
    ;filename extension "CO"
    mvi a,'C'
    sta (filnam + 6)
    mvi a,'O'
    sta (filnam + 7)
    ;is this necessary?
    mvi a,' '
    sta (filnam + 8)
    
    ;make sure length filename >= 1
    lda filnam
    ana a
    jz sgfail
    
    ;create the save file
    call mylnkfil ;fix up directory
    call mysrccom ;search dir for filnam
    cnz mykilcom  ;kill if exists
    call myscnemp ;find empty dir slot
    jz sgfail
    shld dirslot
    lhld bintab ;save bintab since
    push h      ; makhol damages it
    lhld svwnext;needed space
    lxi d,6     ;+size of CO header
    dad d
    mov b,h
    mov c,l     ;bc=amt room to make
    lhld vartab ;hl=at end of CO files
    push h
    call mymakhol
    pop h
    jc sgfail
    ;write CO file header
    ;start address = 0
    xra a
    mov m,a
    inx h
    mov m,a
    inx h
    ;length
    lda (svwnext + 0)
    mov m,a
    inx h
    lda (svwnext + 1)
    mov m,a
    inx h
    ;entry pt = 0
    xra a
    mov m,a
    inx h
    mov m,a
    inx h
    xchg ;save hl in de
    ;restore bintab
    pop h
    push d ;next file location to write
    shld bintab
    ;fill in dir entry
    lhld dirslot
    mvi m,0a0h  ;directory flag
    inx h
    dcx d
    dcx d
    dcx d
    dcx d
    dcx d
    dcx d
    mov m,e     ;file start ptr
    inx h
    mov m,d
    inx h
    ;8-byte file name
    mvi b,8
    lxi d,filnam
sgcpfn:
    ldax d
    mov m,a
    inx d
    inx h
    dcr b
    jnz sgcpfn
    
    call mylnkfil ;fix up directory
    
    ;SECOND PASS: svwrite now actually
    ; writes to the file
    
    ;set up for svwrite
    pop h ;next file location to write
    shld svwnext
    shld svwritefilestart
    mvi a,0c3h ;'jmp' opcode
    sta svwrite
    lxi h,svwrite2
    shld (svwrite+1)
    
    ;set up for svreaddyn, svreaddynend
    lhld dynbase
    shld svrdnext
    
    ;set up for readgam
    lxi h,2f00h ;start of game image
    shld rgnext
    
    ;do second pass
    call dosave
sgpass:
    stc
    ret
sgfail:
    ana a ;clear cy
    ret

;
myscnemp:
    lxi h,0f9afh
    lxi b,0bh
myscnemplp:
    dad b
    mov a,m
    cpi 0ffh
    rz
    add a
    jc myscnemplp
    ora c ;clear zero flag
    ret

    
;helper for savegame
;each time it is called, this version
; of svwrite increments svwnext
;inpars: a
;outpars: none
svwrite1:
    push h
    lhld svwnext
    inx h
    shld svwnext
    pop h
    ret

;helper for savegame
;each time it is called, this version
; of svwrite writes a to adx svwnext in
; save file and increments svwnext
;inpars: a
;outpars: none
svwrite2:
    push h
    lhld svwnext
    mov m,a
    inx h
    shld svwnext
    pop h
    ret
    
;helper for savegame
;writes a to adx hl in save file
svwriteto:
    mov m,a
    ret
    
;helper for savegame
;each time it is called, it reads the
; byte at address svrdnext, increments
; svrdnext and returns the read byte
;inpars: none
;outpar: a is the read byte
svreaddyn:
    push h
    lhld svrdnext
    mov a,m
    inx h
    shld svrdnext
    pop h
    ret
    
;helper for savegame
;checks if svrdnext > dyntop, and
; returns with carry set iff so
;ie checks if most recently read byte
; was not in dyn mem
;inpars: none
;outpar: z flag
svreaddynend:
    push d
    push h
    lhld dyntop
    xchg
    lhld svrdnext
    call cmp16
    pop h
    pop d
    ret
    
;helper for savegame, restoregame
;each time it is called, it reads the
; game image byte at rgnext, increments
; rgnext and returns the read byte
;NOTE we expect that this will only
; ever read game image bytes from the
; first REX block
;inpars: none
;outpar: a is the read byte
readgam:
    push h
    lhld rgnext
    mov a,m
    inx h
    shld rgnext
    pop h
    ret
    
;helper for savegame
;reads hl[bc] where hl is an array of
; big-endian 16-bit words
;inpars: hl base of array
;        bc index into array
;outpar: de the read word
;MODIFIES bc
readrelword:
    push h
    dad b
    dad b
    mov d,m
    inx h
    mov e,m
    pop h
    ret

;worker for savegame
;does one pass of save, using
; - svwrite to write next byte of output file
; - svwriteto to write a specific byte of output file
; - svreaddyn to read next byte of dyn mem
; - svreaddynend to detect if we have read to
;    or past the end of dyn mem
; - readgam to read next byte of game image
;the above functions must be set up
; before this function is called
;inpars: none
;outpar: none
;MODIFIES ALL REGS
dosave:
;;{
    ;;uint16_t *pframe;
    ;;uint16_t wordsused;
    ;;uint8_t  numlocals;
    ;;uint16_t temp16;
    ;;uint8_t  temp8;
    ;;uint16_t *formlengthloc;
    ;;uint16_t *chunklengthloc;
    ;;uint8_t   zerocount;
    
    ;;/**** Start of file ****/
    
    ;;svwrite('F');
    ;;svwrite('O');
    ;;svwrite('R');
    ;;svwrite('M');
    ;;formlengthloc = svwnext;
    ;;svwrite(0); // leave space for form length
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite('I');
    ;;svwrite('F');
    ;;svwrite('Z');
    ;;svwrite('S');
    
    mvi a,'F'
    call svwrite
    mvi a,'O'
    call svwrite
    mvi a,'R'
    call svwrite
    mvi a,'M'
    call svwrite
    lhld svwnext
    shld formlengthloc
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    mvi a,'I'
    call svwrite
    mvi a,'F'
    call svwrite
    mvi a,'Z'
    call svwrite
    mvi a,'S'
    call svwrite
    
    ;;/**** Write IFhd ****/
    
    ;;svwrite('I');
    ;;svwrite('F');
    ;;svwrite('h');
    ;;svwrite('d');
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite(13);
    
    mvi a,'I'
    call svwrite
    mvi a,'F'
    call svwrite
    mvi a,'h'
    call svwrite
    mvi a,'d'
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    mvi a,13
    call svwrite
    
    ;;/* (following fields from game image header) */
    ;;svwrite(release_num[0]);
    ;;svwrite(release_num[1]);
    ;;svwrite(ser_num[0]);
    ;;svwrite(ser_num[1]);
    ;;svwrite(ser_num[2]);
    ;;svwrite(ser_num[3]);
    ;;svwrite(ser_num[4]);
    ;;svwrite(ser_num[5]);
    ;;svwrite(checksum[0]);
    ;;svwrite(checksum[1]);
    ;;/* pc on restore */
    ;;svwrite(zpc[0]);
    ;;svwrite(zpc[1]);
    ;;svwrite(zpc[2]);
    
    lda (roimg+2h) ;release number
    call svwrite
    lda (roimg+3h) ;release number
    call svwrite
    lda (roimg+12h) ;serial number
    call svwrite
    lda (roimg+13h) ;serial number
    call svwrite
    lda (roimg+14h) ;serial number
    call svwrite
    lda (roimg+15h) ;serial number
    call svwrite
    lda (roimg+16h) ;serial number
    call svwrite
    lda (roimg+17h) ;serial number
    call svwrite
    lda (roimg+1ch) ;checksum
    call svwrite
    lda (roimg+1dh) ;checksum
    call svwrite
    lxi h,zpc
    call getl ; zpc in cde
    mov a,c ;initial zpc on restore
    call svwrite
    mov a,d ;initial zpc on restore
    call svwrite
    mov a,e ;initial zpc on restore
    call svwrite
    
    ;;/* write a pad byte */
    ;;svwrite(0);
    xra a
    call svwrite
    
    ;;/**** Write CMem ****/
    
    ;;svwrite('C');
    ;;svwrite('M');
    ;;svwrite('e');
    ;;svwrite('m');
    ;;chunklengthloc = svwnext;
    ;;svwrite(0); // leave space for chunk length
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite(0);
    
    mvi a,'C'
    call svwrite
    mvi a,'M'
    call svwrite
    mvi a,'e'
    call svwrite
    mvi a,'m'
    call svwrite
    lhld svwnext
    shld chunklengthloc
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    
    ;;temp8 = svreaddyn() ^ readgam();
    ;;while (!svreaddynend())
    ;;{
        ;;svwrite(temp8);

    call svreaddyn
    mov b,a
    call readgam
    xra b
    sta temp8
svwhile1:
    call svreaddynend
    jc svendwhile1
    lda temp8
    call svwrite

        ;;if (temp8 == 0)
        ;;{
        
    lda temp8
    ana a
    jnz svelse1
    
            ;;/* count how many additional zeros, up to max 255 */
            ;;zerocount = 0;
            ;;while (1)
            ;;{
            
    xra a
    sta zerocount
svwhile11:
    
                ;;temp8 = svreaddyn() ^ readgam();
                
    call svreaddyn
    mov b,a
    call readgam
    xra b
    sta temp8
    
                ;;if (svreaddynend())
                ;;{
                    ;;break;
                ;;}
                
    call svreaddynend
    jc svendwhile11
    
                ;;if (temp8 != 0)
                ;;{
                    ;;break;
                ;;}
                
    lda temp8
    ana a
    jnz svendwhile11
    
                ;;if (zerocount == 255)
                ;;{
                    ;;break;
                ;;}
                ;;++zerocount;

    lda zerocount
    inr a
    jz svendwhile11
    sta zerocount
    
            ;;}
            
            jmp svwhile11
svendwhile11:

            ;;svwrite(zerocount);
            
            lda zerocount
            call svwrite
            
        ;;}
        ;;else
        ;;{
        
    jmp svendif1
svelse1:

            ;;temp8 = svreaddyn() ^ readgam();
            
    call svreaddyn
    mov b,a
    call readgam
    xra b
    sta temp8
    
        ;;}
        
svendif1:

    ;;}
    
    jmp svwhile1
svendwhile1:
    
    ;;/* write chunk length */
    ;;temp16 = svwnext - chunklengthloc - 4;
    ;;svwriteto(chunklengthloc+2, hi(temp16));
    ;;svwriteto(chunklengthloc+3, lo(temp16));
    
    ;;/* if chunk length was odd, write a pad byte */
    ;;if ((temp16 & 0x0001) != 0)
    ;;{
        ;;svwrite(0);
    ;;}
    
    lhld svwnext
    xchg
    lhld chunklengthloc
    call sub16
    dcx h
    dcx h
    dcx h
    dcx h
    xchg ; de = svwnext - chunklengthloc - 4
    lhld chunklengthloc
    inx h
    inx h
    mov a,d
    call svwriteto
    inx h
    mov a,e
    call svwriteto
    
    mvi a,01h
    ana e
    jz svendif2
    xra a
    call svwrite
svendif2:
    
    ;;/**** Write Stks ****/
    
    ;;svwrite('S');
    ;;svwrite('t');
    ;;svwrite('k');
    ;;svwrite('s');
    ;;chunklengthloc = svwnext;
    ;;svwrite(0); // leave space for chunk length
    ;;svwrite(0);
    ;;svwrite(0);
    ;;svwrite(0);
    
    mvi a,'S'
    call svwrite
    mvi a,'t'
    call svwrite
    mvi a,'k'
    call svwrite
    mvi a,'s'
    call svwrite
    lhld svwnext
    shld chunklengthloc
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    xra a
    call svwrite
    
    ;;pframe = bottom_of_zstk;
    ;;while (1)
    ;;{
    
    lhld zstk
    shld pframe
svwhile2:

        ;;/* return PC */
        ;;svwrite(lo(pframe[0] & 0x0001));
        ;;svwrite(hi(pframe[1]);
        ;;svwrite(lo(pframe[1]);
        
    lhld pframe
    lxi b,0
    call readrelword ;de = pframe[0]
    mvi a,01h
    ana e
    call svwrite
    lhld pframe
    lxi b,1
    call readrelword ;de = pframe[1]
    mov a,d
    call svwrite
    mov a,e
    call svwrite
        
        ;;/* flags (= #locals)*/
        ;;numlocals = pframe[0] >> 1 & 0x000f;
        ;;svwrite(numlocals);
        
    lhld pframe
    lxi b,0
    call readrelword ;de = pframe[0]
    mov a,e
    rrc
    ani 0fh
    sta numlocals
    call svwrite
        
        ;;/* var for res */
        ;;svwrite(hi(pframe[0]));
        
    mov a,d
    call svwrite
        
        ;;/* args supplied */
        ;;switch (lo(pframe[0]) >> 6 & 0x03)
        ;;{
            ;;case 0:
                ;;svwrite(0x00);
                ;;break;
            ;;case 1:
                ;;svwrite(0x01);
                ;;break;
            ;;case 2:
                ;;svwrite(0x03);
                ;;break;
            ;;case 3:
            ;;default:
                ;;svwrite(0x07);
                ;;break;
        ;;}
        
    mov a,e
    rlc
    rlc
    ani 03h
    jz svsw1case0
    dcr a
    jz svsw1case1
    dcr a
    jz svsw1case2
svsw1case3:
svsw1default:
    mvi a,7
    jmp svsw1end
svsw1case2:
    mvi a,3
    jmp svsw1end
svsw1case1:
    mvi a,1
    jmp svsw1end
svsw1case0:
svsw1end:
    call svwrite
        
        ;;/* get wordsused */
        ;;wordsused = pframe[2];
        ;;if (wordsused == 0)
        ;;{
            ;;// last frame
            ;;wordsused = &zstk[zsp] - pframe - 3;
            ;;temp8 = 0; // will exit loop
        ;;}
        ;;else
        ;;{
            ;;temp8 = <nonzero>;
        ;;}
        
    lhld pframe
    lxi b,2
    call readrelword ;de = pframe[2]
    xchg
    shld wordsused
    mov a,h
    ora l
    sta temp8
    jnz wuisset
    lhld zsp
    xchg
    lhld zstk
    dad d
    dad d
    xchg
    lhld pframe
    call sub16
    ;need to divide by 2
    xra a ;clear cy
    mov a,h
    rar
    mov h,a
    mov a,l
    rar
    mov l,a
    dcx h
    dcx h
    dcx h
    shld wordsused
wuisset:

        ;;/* # eval stack words */
        ;;temp16 = wordsused - 1; // #locals + #eval
        ;;temp16 -= numlocals; // #eval
        ;;svwrite(hi(temp16));
        ;;svwrite(lo(temp16));
        
    lhld wordsused
    dcx h
    xchg
    mvi h,0
    lda numlocals
    mov l,a
    call sub16
    mov a,h
    call svwrite
    mov a,l
    call svwrite
        
        ;;/* locals and eval stack*/
        ;;temp16 = wordsused - 1 + 4;
        ;;for (i = 4; i < temp16; ++i)
        ;;{
            ;;svwrite(hi(pframe[i]));
            ;;svwrite(lo(pframe[i]));
        ;;}
        
    lhld wordsused
    inx h
    inx h
    inx h
    shld temp16
    lxi b,4
svfor1:
    mov d,b
    mov e,c
    lhld temp16
    call cmp16
    jnc svendfor1
    lhld pframe
    call readrelword; de = pframe[bc]
    mov a,d
    call svwrite
    mov a,e
    call svwrite
    inx b
    jmp svfor1
svendfor1:
        
        ;;/* next frame */
        ;;if (temp8 == 0)
        ;;{
            ;;break;
        ;;}
        ;;else
        ;;{
            ;;pframe += wordsused + 3;
        ;;}
        
    lda temp8
    ana a
    jz svendwhile2
    lhld wordsused
    dad h
    lxi d,6
    dad d
    xchg
    lhld pframe
    dad d
    shld pframe
        
    ;;}
    
    jmp svwhile2
svendwhile2:
    
    ;;/* write chunk length - note it's always even */
    ;;temp16 = svwnext - chunklengthloc - 4;
    ;;writeTo(chunklengthloc+2, hi(temp16));
    ;;writeTo(chunklengthloc+3, lo(temp16));
    
    lhld svwnext
    xchg
    lhld chunklengthloc
    call sub16
    dcx h
    dcx h
    dcx h
    dcx h
    xchg ; de = svwnext - chunklengthloc - 4
    lhld chunklengthloc
    inx h
    inx h
    mov a,d
    call svwriteto
    inx h
    mov a,e
    call svwriteto

    ;;/**** End of file ****/
    
    ;;/* write form length */
    ;;temp16 = svwnext - formlengthloc - 4;
    ;;writeTo(formlengthloc+2, hi(temp16));
    ;;writeTo(formlengthloc+3, lo(temp16));

    ;;/* if form length was odd, write a pad byte */
    ;;if ((temp16 & 0x0001) != 0)
    ;;{
        ;;svwrite(0);
    ;;}
    
    lhld svwnext
    xchg
    lhld formlengthloc
    call sub16
    dcx h
    dcx h
    dcx h
    dcx h
    xchg ; de = svwnext - formlengthloc - 4
    lhld formlengthloc
    inx h
    inx h
    mov a,d
    call svwriteto
    inx h
    mov a,e
    call svwriteto
    
    mvi a,01h
    ana e
    jz svendif3
    xra a
    call svwrite
svendif3:

;;}

    ret


;restores the game
;inpars: none
;outpars:
; cy=1 if successful
; cy=0 if not
restoregame:
    ;FIRST PASS: dorestore is run with a
    ; version of rswrite which doesn't
    ; actually write to dyn mem - it
    ; just increments the write pointer
    ; this lets us verify the file first
    
    mvi a,0c3h ;'jmp' opcode
    sta rswrite
    lxi h,rswrite2
    shld (rswrite+1)
    
    ;get filename
    mvi a,(fnpromptend-fnprompt)
    lxi h,fnprompt
    call printstring
    call getfilename
    
    ;filename extension "CO"
    mvi a,'C'
    sta (filnam + 6)
    mvi a,'O'
    sta (filnam + 7)
    ;is this necessary?
    mvi a,' '
    sta (filnam + 8)
    
    ;make sure length filename >= 1
    lda filnam
    ana a
    jz sgfail
    
    ;find the save file
    call mylnkfil ;fix up directory
    call mysrccom ;search dir for filnam
    jz rgfail ;file doesn't exist
    
    ;set up for rsread, rsreadremaining
    xchg
    lxi d,6
    dad d
    shld rsrdnext
    push h
    dcx h
    dcx h
    dcx h
    dcx h
    mov e,m
    inx h
    mov d,m
    inx h
    inx h
    inx h
    dad d
    shld rsrdend
    push h
    
    ;set up for rswrite, writeremaining
    lhld dynbase
    shld rswrnext
    lhld dyntop
    shld rswrend
    
    ;set up for readgam
    lxi h,2f00h ;start of game image
    shld rgnext
    
    ;do first pass
    call dorestore
    ana a
    jnz rgpass1ok
    pop h
    pop h
    jmp rgfail
    
rgpass1ok:
    
    ;SECOND PASS: rswrite now actually
    ; writes to the dyn mem
    
    mvi a,0c3h ;'jmp' opcode
    sta rswrite
    lxi h,rswrite2
    shld (rswrite+1)
    
    ;set up for rsread, rsreadremaining
    pop h
    shld rsrdend
    pop h
    shld rsrdnext
    
    ;set up for rswrite, writeremaining
    lhld dynbase
    shld rswrnext
    lhld dyntop
    shld rswrend
    
    ;set up for readgam
    lxi h,2f00h ;start of game image
    shld rgnext
    
    ;do second pass
    call dorestore
    ana a
    jz rgfail
    
    ;restore zpc and zsp
    lxi h,restorezpc
    call getl
    lxi h,zpc
    call putl
    lhld restorezsp
    shld zsp
    lhld callerszlp
    shld zlp
    
rgpass:
    stc
    ret
rgfail:
    ana a ;clear cy
    ret
    
;helper for restoregame
;each time it is called, it reads the
; file byte at address rsrdnext,
; increments rsrdnext and returns the
; read byte
;inpars: none
;outpar: a is the read byte
rsread:
    push h
    lhld rsrdnext
    mov a,m
    inx h
    shld rsrdnext
    pop h
    ret

;helper for restoregame
;returns the number of bytes not yet
; read from file
;inpars: none
;outpar: de the number bytes remaining
rsreadremaining:
    push h
    lhld rsrdend
    xchg
    lhld rsrdnext
    call sub16
    xchg
    pop h
    ret

;helper for restoregame
;each time it is called, this version
; of rswrite increments rswrnext
;inpars: a is the value to write
;outpar: none
rswrite1:
    push h
    lhld rswrnext
    inx h
    shld rswrnext
    pop h
    ret
    
;helper for restoregame
;each time it is called, it writes a to
; the byte at address rswrnext in dyn
; mem, then increments rswrnext
;inpars: a is the value to write
;outpar: none
rswrite2:
    push h
    lhld rswrnext
    mov m,a
    inx h
    shld rswrnext
    pop h
    ret
    
;helper for restoregame
;returns the number of bytes not yet
; written to dyn mem
;inpars: none
;outpar: de the number bytes remaining
writeremaining:
    push h
    lhld rswrend
    xchg
    lhld rswrnext
    call sub16
    xchg
    pop h
    ret

;helper for restoregame
;writes de to hl[bc] where hl is an
; array of big-endian 16-bit words
;inpars: hl base of array
;        bc index into array
;        de the value to write
;outpar: none
;MODIFIES bc
writerelword:
    push h
    dad b
    dad b
    mov m,d
    inx h
    mov m,e
    pop h
    ret

;worker for restoregame
;does one pass of restore, using
; - rsread to read next byte from input file
; - rsreadremaining to get #bytes left to read from file
; - rswrite to write next byte of dyn mem
; - writeremaining to get #bytes left to write to dyn mem
; - readgam to read next byte of game image
;the above functions must be set up
; before this function is called
;inpars: none
;outpars: a = 0 if any error, else a = 1
;         on success, new values in
;         callerszlp, restorezpc
;MODIFIES ALL REGS
dorestore:
;{
    ;;uint16_t *pframe;
    ;;uint8_t numlocals;
    ;;uint16_t numevalwords;
    ;;uint16_t callerszlp;
    ;;uint16_t formlength;
    ;;uint16_t *formstartloc;
    ;;uint16_t chunklength;
    ;;uint16_t *chunkstartloc;
    ;;uint32_t restorezpc;
    ;;uint8_t gotcmem;
    ;;uint8_t gotstks;
    ;;uint8_t temp8;
    ;;uint16_t temp16;
    ;;uint8_t zerocount;
    
    ;;/* Check if file big enough for at least
        ;;FORM (len) IFZS IFhd (len) (13 bytes contents of IFhd) (pad byte) */
    ;;if (rsreadremaining() < 34) return 0;
    call rsreadremaining ;remaining #bytes in de
    lxi h,34
    call cmp16
    jc rsfail
    
    ;;/**** Start of file ****/
    
    ;;if (rsread() != 'F') return 0;
    ;;if (rsread() != 'O') return 0;
    ;;if (rsread() != 'R') return 0;
    ;;if (rsread() != 'M') return 0;
    ;;if (rsread() != 0) return 0;
    ;;if (rsread() != 0) return 0;
    
    ;;formlength = rsread()<<8 | rsread();
    ;;formstartloc = cur_read_loc;
    
    call rsread
    cpi 'F'
    jnz rsfail
    call rsread
    cpi 'O'
    jnz rsfail
    call rsread
    cpi 'R'
    jnz rsfail
    call rsread
    cpi 'M'
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    mov h,a
    call rsread
    mov l,a
    shld formlength
    lhld rsrdnext
    shld formstartloc
    
    ;;if (rsread() != 'I') return 0;
    ;;if (rsread() != 'F') return 0;
    ;;if (rsread() != 'Z') return 0;
    ;;if (rsread() != 'S') return 0;
    
    call rsread
    cpi 'I'
    jnz rsfail
    call rsread
    cpi 'F'
    jnz rsfail
    call rsread
    cpi 'Z'
    jnz rsfail
    call rsread
    cpi 'S'
    jnz rsfail
    
    ;;/**** Read IFhd - it must be the first chunk ****/
    
    ;;if (rsread() != 'I') return 0;
    ;;if (rsread() != 'F') return 0;
    ;;if (rsread() != 'h') return 0;
    ;;if (rsread() != 'd') return 0;
    ;;if (rsread() != 0) return 0;
    ;;if (rsread() != 0) return 0;
    ;;if (rsread() != 0) return 0;
    ;;if (rsread() != 13) return 0;
    
    call rsread
    cpi 'I'
    jnz rsfail
    call rsread
    cpi 'F'
    jnz rsfail
    call rsread
    cpi 'h'
    jnz rsfail
    call rsread
    cpi 'd'
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    cpi 13
    jnz rsfail
    
    ;;/* (following fields from game image header) */
    ;;if (rsread() != release_num[0]) return 0;
    ;;if (rsread() != release_num[1]) return 0;
    ;;if (rsread() != ser_num[0]) return 0;
    ;;if (rsread() != ser_num[1]) return 0;
    ;;if (rsread() != ser_num[2]) return 0;
    ;;if (rsread() != ser_num[3]) return 0;
    ;;if (rsread() != ser_num[4]) return 0;
    ;;if (rsread() != ser_num[5]) return 0;
    ;;if (rsread() != checksum[0]) return 0;
    ;;if (rsread() != checksum[1]) return 0;
    ;;/* pc on restore */
    ;;restorezpc = rsread()<<16 | rsread()<<8 | rsread();
    
    call rsread
    mov b,a
    lda (roimg+2h) ;release number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+3h) ;release number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+12h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+13h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+14h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+15h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+16h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+17h) ;serial number
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+1ch) ;checksum
    cmp b
    jnz rsfail
    call rsread
    mov b,a
    lda (roimg+1dh) ;checksum
    cmp b
    jnz rsfail
    call rsread ; zpc on restore
    mov c,a
    ani 0feh
    jnz rsfail
    call rsread
    mov d,a
    call rsread
    mov e,a
    lxi h,restorezpc
    call putl
    
    ;;/* read pad byte */
    ;;rsread();
    
    call rsread
    
    ;;/* read 'CMem' and 'Stks' chunks - either order */
    ;;/* NOTE currently no other chunks are supported,
       ;;and if they are present they must be after the
       ;;'CMem' and 'Stks' chunks */
    ;;gotcmem = 0;
    ;;gotstks = 0;
    
    xra a
    sta gotcmem
    sta gotstks
    
    ;;while (!gotcmem || !gotstks)
    ;;{
    
rswhile1:
    lda gotcmem
    mov b,a
    lda gotstks
    ana b
    jnz rsendwhile1
    
        ;;/* check if file big enough for at least
            ;;next chunk name and length. */
        ;;if (rsreadremaining() < 8) return 0;
        
        call rsreadremaining ;remaining #bytes in de
        lxi h,8
        call cmp16
        jc rsfail
        
        ;;/* read next chunk */
        ;;switch (rsread())
        ;;{
        
    call rsread
    cpi 'C'
    jz rssw1casec
    cpi 'S'
    jz rssw1cases
    jmp rsfail
    
            ;;case 'C': // 'CMem' chunk
            
rssw1casec:

                ;;if (gotcmem) return 0;
                ;;if (rsread() != 'M') return 0;
                ;;if (rsread() != 'e') return 0;
                ;;if (rsread() != 'm') return 0;
                ;;if (rsread() != 0) return 0;
                ;;if (rsread() != 0) return 0;
                
                ;;chunklength = rsread()<<8 | rsread();
                ;;chunkstartloc = cur_read_loc;
                
    lda gotcmem
    ana a
    jnz rsfail
    call rsread
    cpi 'M'
    jnz rsfail
    call rsread
    cpi 'e'
    jnz rsfail
    call rsread
    cpi 'm'
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    mov h,a
    call rsread
    mov l,a
    shld chunklength
    lhld rsrdnext
    shld chunkstartloc
                
                ;;while (writeremaining() && rsreadremaining())
                ;;{
                
rswhile2:
    call writeremaining ;remaining #bytes in de
    mov a,d
    ora e
    jz rsendwhile2
    call rsreadremaining ;remaining #bytes in de
    mov a,d
    ora e
    jz rsendwhile2
    
                    ;;temp8 = rsread();
                    ;;rswrite(temp8 ^ readgam());
                    ;;if (temp8 == 0 && rsreadremaining())
                    ;;{
                    
    call rsread
    mov b,a
    call readgam
    xra b
    call rswrite
    mov a,b
    ana a
    jnz rsendif1
    call rsreadremaining ;remaining #bytes in de
    mov a,d
    ora e
    jz rsendif1
    
                        ;;/* do additional zeros */
                        ;;zerocount = rsread();
                        ;;if (zerocount > writeremaining())
                        ;;{
                            ;;return 0;
                        ;;}
                        
    call rsread
    mov b,a ;(b is zerocount)
    call writeremaining ;remaining #bytes in de
    inr d
    dcr d
    jnz rsendif2
    mov a,e
    sub b
    jc rsfail
rsendif2:
                        
                        ;;while(zerocount)
                        ;;{
                            ;;rswrite(readgam());
                            ;;--zerocount;
                        ;;}
                        
rswhile3:
    inr b
    dcr b
    jz rsendwhile3
    call readgam
    call rswrite
    dcr b
    jmp rswhile3
rsendwhile3:
                        
                    ;;}
                    
rsendif1:
                    
                ;;}
                
    jmp rswhile2
rsendwhile2:
                
                ;;/* copy rest of game image */
                ;;while (writeremaining())
                ;;{
                    ;;rswrite(readgam());
                ;;}
                
rswhile4:
    call writeremaining ;remaining #bytes in de
    mov a,d
    ora e
    jz rsendwhile4
    call readgam
    call rswrite
    jmp rswhile4
rsendwhile4:
                
                ;;gotcmem = 1;
                ;;break;
                
    mvi a,1
    sta gotcmem
    jmp rssw1end
                
            ;;case 'S': // 'Stks' chunk
            
rssw1cases:

                ;;if (gotstks) return 0;
                ;;if (rsread() != 't') return 0;
                ;;if (rsread() != 'k') return 0;
                ;;if (rsread() != 's') return 0;
                ;;if (rsread() != 0) return 0;
                ;;if (rsread() != 0) return 0;
                
                ;;chunklength = rsread() << 8 | rsread();
                ;;chunkstartloc = cur_read_loc;
                
    lda gotstks
    ana a
    jnz rsfail
    call rsread
    cpi 't'
    jnz rsfail
    call rsread
    cpi 'k'
    jnz rsfail
    call rsread
    cpi 's'
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    ana a
    jnz rsfail
    call rsread
    mov h,a
    call rsread
    mov l,a
    shld chunklength
    lhld rsrdnext
    shld chunkstartloc
    ; make sure chunk isn't too big
    ; for our stack
    ; this works because although the
    ; format is different, our stack
    ; frames have the same size as the
    ; frames in the save file
    lxi d,zstksize
    lhld chunklength
    call cmp16
    jc rsfail
                
                ;;pframe = bottom_of_zstk;
                ;;callerszlp = 0;
                ;;while (1)
                ;;{
                
    lhld zstk
    shld pframe
    lxi h,0
    shld callerszlp
rswhile5:
                
                    ;;/* check if enough for minimum stack frame size left */
                    ;;if (rsreadremaining() < 8)
                    ;;{
                        ;;return 0;
                    ;;}
                    
    call rsreadremaining ;remaining #bytes in de
    lxi h,8
    call cmp16
    jc rsfail
        
                    ;;/* return PC */
                    ;;temp8 = rsread();
                    ;;if (temp8 & 0xfe)
                    ;;{
                        ;;return 0;
                    ;;}
                    ;;pframe[0] = temp8;
                    ;;pframe[1] = rsread() << 8 | rsread();
                    
    ;for now keep pframe[0] in temp16
    call rsread
    mov l,a
    ani 0feh
    jnz rsfail
    mvi h,0
    shld temp16
    call rsread
    mov d,a
    call rsread
    mov e,a
    lxi b,1
    lhld pframe
    call writerelword
                    
                    ;;/* flags (= #locals) */
                    ;;numlocals = rsread();
                    ;;if (numlocals & 0xf0)
                    ;;{
                        ;;return 0;
                    ;;}
                    ;;pframe[0] |= numlocals << 1;
                    
                    ;;/* var for res */
                    ;;pframe[0] |= rsread() << 8;
                    
    call rsread
    sta numlocals
    ani 0f0h
    jnz rsfail
    lda numlocals
    add a
    mov b,a
    lda temp16
    ora b
    sta temp16
    call rsread
    sta (temp16 + 1)
                    
                    ;;/* args supplied */
                    ;;switch (rsread())
                    ;;{
                        ;;case 0:
                            ;;break;
                        ;;case 1:
                            ;;pframe[0] |= 0x0040;
                            ;;break;
                        ;;case 3:
                            ;;pframe[0] |= 0x0080;
                            ;;break;
                        ;;case 7:
                            ;;pframe[0] |= 0x00c0;
                            ;;break;
                        ;;default:
                            ;;return 0;
                    ;;}
                    
    call rsread
    ana a
    jz rssw2end
    cpi 1
    mvi b,040h
    jz rssw2set
    cpi 3
    mvi b,080h
    jz rssw2set
    cpi 7
    mvi b,0c0h
    jnz rsfail
rssw2set:
    lda temp16
    ora b
    sta temp16
rssw2end:

    ;now we can save pframe[0]
    lhld temp16
    xchg
    lhld pframe
    lxi b,0
    call writerelword
                    
                    ;;/* # eval stack words */
                    ;;numevalwords = rsread() << 8 | rsread();
                    
    call rsread
    mov h,a
    call rsread
    mov l,a
    shld numevalwords
                    
                    ;;pframe[3] = callerszlp;
                    ;;callerszlp = pframe - bottom_of_zstk + 3; // note WORD pointers
                    
    lhld callerszlp
    xchg
    lhld pframe
    lxi b,3
    call writerelword
    lhld pframe
    xchg
    lhld zstk
    call sub16
    ;need to divide by 2
    xra a ;clear cy
    mov a,h
    rar
    mov h,a
    mov a,l
    rar
    mov l,a
    lxi d,3
    dad d
    shld callerszlp
                    
                    ;;/* check if enough left for locals and eval stack words */
                    ;;if (rsreadremaining() < 2*(numlocals + numevalwords))
                    ;;{
                        ;;return 0;
                    ;;}
                    
    lda numlocals
    mov e,a
    mvi d,0
    lhld numevalwords
    dad d
    push h ;save numlocals + numevalwords
    dad h
    call rsreadremaining ;remaining #bytes in de
    call cmp16
    pop h
    jc rsfail
                    
                    ;;/* local vars and eval stack */
                    ;;temp16 = 4 + numlocals + numevalwords;
                    
    lxi d,4
    dad d
    shld temp16
                    
                    ;;for (i = 4; i < temp16; ++i)
                    ;;{
                        ;;pframe[i] = rsread() << 8 | rsread();
                    ;;}
    
    lxi b,4
rsfor1:
    mov d,b
    mov e,c
    lhld temp16
    call cmp16
    jnc rsendfor1
    call rsread
    mov d,a
    call rsread
    mov e,a
    lhld pframe
    call writerelword
    inx b
    jmp rsfor1
rsendfor1:
                    
                    ;;/* WORDS USED */
                    ;;if (cur_read_loc - chunkstartloc == chunklength)
                    ;;{
                        ;;pframe[2] = 0;
                        ;;break;
                        
    lhld rsrdnext
    xchg
    lhld chunkstartloc
    call sub16
    xchg
    lhld chunklength
    call cmp16
    jnz rselse3
    ; save final zsp
    ;
    mov h,b
    mov l,c
    shld restorezsp
    ;
    lxi b,2
    lxi d,0
    lhld pframe
    call writerelword
    jmp rsendwhile5
                        
                    ;;}
                    ;;else
                    ;;{
                    
rselse3:
                    
                        ;;pframe[2] = numlocals + numevalwords + 1;
                        
    lda numlocals
    mov e,a
    mvi d,0
    lhld numevalwords
    dad d
    inx h
    xchg
    lxi b,2
    lhld pframe
    call writerelword
                        
                    ;;}
                    
rsendif3:
                    
                    ;;pframe += temp16;

    lhld temp16
    xchg
    lhld pframe
    dad d
    dad d ;add 2x since word ptr
    shld pframe

                ;;}
                
    jmp rswhile5
rsendwhile5:
                
                ;;gotstks = 1;
                ;;break;
                
    mvi a,1
    sta gotstks
    jmp rssw1end
                
            ;;default:
                ;;return 0;
        ;;}

rssw1end:
        
        ;;/* check that chunk length was correct */
        ;;if (chunklength != cur_read_loc - chunkstartloc)
        ;;{
            ;;return 0;
        ;;}
        
    lhld rsrdnext
    xchg
    lhld chunkstartloc
    call sub16
    xchg
    lhld chunklength
    call cmp16
    jnz rsfail
    
        ;;/* if chunk length was odd, read pad byte */
        ;;if (chunklength & 0x0001)
        ;;{
            ;;rsread();
        ;;}
        
    mov a,l
    ani 01h
    cnz rsread
    ;;}
    
    jmp rswhile1
rsendwhile1:
    
    ;;/**** End of file ****/
    
    ;;/* check that form length was not exceeded */
    ;;if (formlength < cur_read_loc - formstartloc)
    ;;{
        ;;return 0;
    ;;}
    
    lhld rsrdnext
    xchg
    lhld formstartloc
    call sub16
    xchg
    lhld formlength
    xchg
    jc rsfail
    
    ;;return 1;
    
rspass:
    mvi a,1
    ret
rsfail:
    mvi a,0
    ret
    
;;}


;verifies the game image checksum
;(only returns if successful)
;sum of bytes from 40h to end of image
; modulo 10000h
;MODIFIES ALL REGS
verify:
    ; store length of image as
    ;  given in header in temp32
    lxi h,1ah
    call zgtwlw
    mov a,h
    ora l
    rz
    mvi c,0
    mov d,h
    mov e,l
    call ahl24 ; cde = 2*(word at 1ah)
    lxi h,-40h
    call ashl24 ; - 40h
    lxi h,temp32
    call putl ;stored in temp32
    ; init checksum to 0
    lxi h,0
    push h ;keep on stack
    ;
    ;first block
    ;max #img bytes in block:
    lxi h,(7f00h-12032-40h)
    call vermin ;bc=#bytes to process
    pop h
    lxi d,(12032+40h);start at byte 40h
                     ; of game image
    lda rexblock
    call verifyblock
    push h
    ;
    ;second block
    ;max #img bytes in block:
    lxi h,(7f00h-256)
    call vermin ;bc=#bytes to process
    pop h
    lxi d,256 ;game start in block
    lda (rexblock+1)
    call verifyblock
    push h
    ;
    ;third block
    ;max #img bytes in block:
    lxi h,(7f00h-256)
    call vermin ;bc=#bytes to process
    pop h
    lxi d,256 ;game start in block
    lda (rexblock+2)
    call verifyblock
    push h
    ;
    ;fourth block
    ;max #img bytes in block:
    lxi h,(7f00h-256)
    call vermin ;bc=#bytes to process
    pop h
    lxi d,256 ;game start in block
    lda (rexblock+3)
    call verifyblock
    push h
    ;
    ;fifth block
    ;max #img bytes in block:
    lxi h,(7f00h-256)
    call vermin ;bc=#bytes to process
    pop h
    lxi d,256 ;game start in block
    lda (rexblock+4)
    call verifyblock
    push h
    ;
    ;calc'd checksum is on stack
    lxi h,1ch
    call zgtwlw ;hl = expected checksum
    pop d       ;de = calc'd checksum
    call cmp16s
    cnz error
    ret
;
;vermin: helper for verify
;temp32 contains 24-bit #image bytes
; remaining to process
;determines whether temp32 >= hl, then:
; if temp32 >= hl:
;   subtracts hl from temp32
;   puts hl in bc
;   returns
; else:
;   puts temp32 in bc
;   sets temp32 to 0
;   returns
;MODIFIES ALL REGS
vermin:
    push h
    lxi h,temp32
    call getl
    pop h
    mov a,c
    ana a
    jnz t32ge
    call cmp16
    jnc t32ge
    ;hl > temp32
    push d
    lxi d,0 ;cde = 0
    lxi h,temp32
    call putl
    pop b
    ret
t32ge:
    ;temp32 >= hl
    push h
    mov a,e
    sub l
    mov e,a
    mov a,d
    sbb h
    mov d,a
    mov a,c
    sbi 0
    mov c,a
    lxi h,temp32
    call putl
    pop b
    ret


;reads and parses a command from the
; terminal
; i.e. handles the sread operation.
;reads into text buf
;parses into parse buf
;inpars: hl is z-mem adx of text buf
;        de is z-mem adx of parse buf
;outpar: none
;MODIFIES ALL REGS
rdline:
    call moreend ;flush buffered output
    call getline
    call parsln
    call morebegin
    ret
    
;parses a command from the terminal
;inpars: hl is z-mem adx of text buf
;        de is z-mem adx of parse buf
;outpar: none
;MODIFIES ALL REGS
parsln:
    push d
    shld txtbuf
    xchg
    call zgtblw ;max #entries for parse buf
    sta pbentriesleft
    inx h
    inx h
    shld parsebufpos;adx next p.buf entry
    mvi b,1 ;b=index first t.buf letter
prslp:
    ;make sure we have room for another
    ; parse buf entry
    lda pbentriesleft
    ana a
    jz eprslp
    ;b=idx one ch past end of prev word
    ;find start of next word, keep
    ; index in b
swslp:
    mov a,b
    call gttbb
    inr b
    cpi ' '
    jz swslp ;skip leading whitespace
    dcr b
    cpi 0 ;marks end of string
    jz eprslp
    ;now b indexes start of next word
    ;make c index one ch past end of wd
    mov c,b
    inr c
    call issep
    jz prslu ;separator IS a word
prsfe:
    mov a,c
    call gttbb
    cpi ' '
    jz prslu
    cpi 0
    jz prslu
    call issep
    jz prslu
    inr c
    jmp prsfe
prslu: ;look up word in dictionary
    lhld txtbuf
    mov a,b
    call aahl;hl=zmem adx start of word
    mov a,c
    sub b ;a=word length
    call dictlookup
    ;write the parse buf entry
    xchg
    lhld parsebufpos
    call zptwlw ;ptr to dict. entry
    inx h
    inx h
    mov a,c
    sub b
    call zptblw ;word length
    inx h
    mov a,b
    call zptblw ;index word in text buf
    inx h
    shld parsebufpos
    lda pbentriesleft
    dcr a
    sta pbentriesleft
    mov b,c
    jmp prslp
eprslp:
    ;byte 1 of parse buf gets #words
    pop d ;start of parse buf
    lhld parsebufpos
    ;hl - de
    mov a,l
    sub e
    mov l,a
    mov a,h
    sbb d
    mov h,a
    ; - 2
    dcx h ;since 1st 2 bytes of parse
    dcx h ; buf aren't entry bytes
    ;divide by 4=bytes per entry
    ana a ;cy=0
    mov a,h
    rar
    mov h,a
    mov a,l
    rar
    mov l,a
    ana a ;cy=0
    mov a,h
    rar
    mov a,l
    rar
    ;now a is #words
    inx d
    xchg
    call zptblw
    ret
;
;helper for parsln
;gets the byte at zmem adx txtbuf+a
;inpars: (adx in txtbuf), offset a
;outpar: byte in a
gttbb:
    push h
    lhld txtbuf
    call aahl ;hl=zmem adx of the byte
    call zgtblw
    pop h
    ret
;
;helper for parsln
;checks if a is a "word-separator" char
;inpar: char in a
;outpar: z=1 iff a is a "word-sep."
issep:
    push b
    push h
    mov b,a
    mvi c,0
    lxi h,zdseps
seplp:
    mov a,m
    cmp c
    jz ntsepr
    cmp b
    jz issepr
    inx h
    jmp seplp
ntsepr:
    mvi a,1
    ana a ;z=0
issepr:
    pop h
    pop b
    ret
    
;searches for word in the dictionary
;word MUST be lowercase
;NOTE currently doesn't support input
; characters other than standard ascii
; (32 to 126) -> only these are allowed
; in the word
;inpars: 0hl zmem adx of the word
;        a is length of the word
;outpar: 0hl is zmem adx of dict.entry 
;         =0 if not in dictionary
dictlookup:
    push b
    push d
    ana a
    jz dntfnd
    ;convert word to zchars (up to 6),
    ; put in dictword, pad with 5s as ndd
    xchg
    lxi h,0505h
    shld dictword      ; pad with 5s
    shld (dictword+2)  ; pad with 5s
    shld (dictword+4)  ; pad with 5s
    lxi h,dictword
    sta wdcharsleft ;chars left in word
    mvi a,6 ; space left in dictword
    sta dictwdcharsleft
    push h  ; dest on stack
    xchg    ; src in hl
dcnvlp:
    call zgtblw
    inx h
    call tozchar
    xthl ; dest in hl, src on stack
    mov m,d ; 1st zchar to dest
    lda dictwdcharsleft
    mov d,a ; space left in dictword
    dcr d
    jz dwdend
    inx h
    mov a,e ; 2nd zchar if !0
    ana a
    jz donzch
    mov m,a
    dcr d
    jz dwdend
    inx h
    mov a,b ; 3rd zchar if !0
    ana a
    jz donzch
    mov m,a
    dcr d
    jz dwdend
    inx h
    mov a,c ; 4th zchar if !0
    ana a
    jz donzch
    mov m,a
    dcr d
    jz dwdend
    inx h
donzch:
    xthl ; dest on stack, src in hl
    mov a,d ; space left in dictword
    sta dictwdcharsleft
    lda wdcharsleft
    dcr a
    sta wdcharsleft
    jnz dcnvlp
dwdend:
    pop h ; whatever we left on stack
    ;done converting word to zchars
    ;pack them into z-encoded packedword
    lxi h,dictword
    lxi d,packedword
    call dictpack
    call dictpack
    lda (packedword+2) ;MSbit of second
    ori 80h        ;packed word must be
    sta (packedword+2) ;1 indicating last
    ;search for encoded word packedword
    ; in dictionary
    ;dictionary words are in increasing
    ; order
    ;do binary search
    ;index of top in bc
    ;index of bottom in de
    lhld zdnumentries ; # dict entries
    mov b,h
    mov c,l;(bc one past last entry)
    lxi d,0
binsrc:
    ;if top == bottom+1, didn't find it
    mov a,b
    cmp d
    jnz nxtsrc
    mov a,c
    sub e
    dcr a
    jz dntfnd
nxtsrc:
    ;get index of middle entry
    mov h,b
    mov l,c
    dad d
    mov a,h
    rar
    mov h,a
    mov a,l
    rar
    mov l,a ; hl = (top+bottom)/2
    ;compare dictionary entry hl with
    ; z-encoded word in packedword
    ; (dict word @hl) - (packedword)
    call dictcomp
    jz dfnd
    jnc dgtt
    ; (dict word @hl) < (packedword)
    ; move bottom up
    xchg
    jmp binsrc
dgtt:
    ; (middle dict word) > (packedword)
    ; move top down
    mov b,h
    mov c,l
    jmp binsrc
dntfnd: ; not found in dictionary
    lxi h,0
    pop d
    pop b
    ret
dfnd: ; found word in dictionary and
    ;hl is its zmem address
    pop d
    pop b
    ret
;
;helper for dictlookup
;compares the z-encoded word in packedword
; to the dictionary word at index hl
;i.e. sets flags with
; (dict word hl) - (packedword)
;inpars: (packedword) and index hl
;outpar: flags
;        if equal, hl is replaced by
;         zmem address of the dict word
;        else hl unchanged
dictcomp:
    push b
    push d
    push h
    ;the dict. word is at address
    ; zdictbase + zdentrylength*index
    lda zdentrylength
    mov d,a
    mov b,h
    mov c,l ; index in bc
    ;NOTE dict. in lower 64k zmem
    ; so result should fit in 16 bits
    call mul816u ; hl = zdentrylength*index
    xchg
    lhld zdictbase
    dad d ; hl = zdictbase + zdentrylength*index
    ;compare
    lxi d,packedword ;hl->dict word, de->packedword
    mvi c,4
dcmplp:
    call zgtblw
    xchg
    cmp m
    xchg
    jnz dcmpne
    inx h
    inx d
    dcr c
    jnz dcmplp
    ;they were equal
    push psw
    dcx h
    dcx h
    dcx h
    dcx h ;hl=zmem adx of dict word
    pop psw
    pop d ;discard old hl
    pop d
    pop b
    ret
dcmpne:
    ;they were not equal
    pop h
    pop d
    pop b
    ret
;
;helper for dictlookup
;converts non-uppercase zscii char
; to zchar(s)
;inpar: zscii char in a
;outpars:
;  1st zchar in d
;  2nd zchar in e (0 if < 2)
;  3rd zchar in b (0 if < 3)
;  4th zchar in c (0 if < 4)
tozchar:
    push h
    mov d,a ; save char in d
    cpi 'a'
    jc notlet
    cpi ('z'+1)
    jnc notlet
    ;it's a lowercase letter, so in
    ; alphabet 0
    adi ((6 - 'a')*256/256); to zchar
    mov d,a
    mvi e,0
    lxi b,0
    pop h
    ret
notlet:
    ;try to find in alphabet 2
    lxi h,(za2+7)
    mvi e,25 ; max # chars to check
a2lp:
    mov a,m
    cmp d
    jz ina2
    inx h
    dcr e
    jnz a2lp
    jmp nota2
    ;it's in alphabet 2
ina2:
    lxi d,za2
    xchg
    call sub16
    mvi d,5 ; shift alphabet 2
    mov e,l
    lxi b,0
    pop h
    ret
nota2:    
    ;not in alphabet 2
    ;put all 8 zscii bits in 2 zchars
    ; h and l
    mov a,d
    lxi d,0506h ;5: shift alphabet 2
                ;6: zscii in next 2 zch
    mov c,a
    rlc
    rlc
    rlc
    ani 07h
    mov b,a
    mov a,c
    ani 1fh
    mov c,a
    pop h
    ret
;
;helper for dictlookup
;packs zchars for z-encoding word
;inpars: hl -> str of 3 zchars
;        de -> where to pack them
;outpar: hl advanced past reads
;        de advanced past writes
dictpack:
    push b
    mov b,m ; 1st zchar
    inx h
    mov c,m ; 2nd zchar
    inx h
    mov a,b ; 000ABCDE
    rlc
    rlc
    mov b,a ; 0ABCDE00
    mov a,c ; 000FGHIJ
    rrc
    rrc
    rrc
    mov c,a ; HIJ000FG
    ani 03h ; 000000FG
    ora b
    stax d          ; 0ABCDEFG
    inx d
    mov b,m ; 3rd zchar 000KLMNP
    inx h
    mov a,c ; HIJ000FG
    ani 0e0h; HIJ00000
    ora b   ; HIJKLMNP
    stax d
    inx d
    pop b
    ret

    


;takes care of the remove_obj operation
;inpar: (op1 is object)
;outpar: none
;DESTROYS ALL REGS
remobj:
    ;get parent
    lda op1
    mov b,a ; save obj in b
    call goepar
    ana a
    jz remps ; no parent
    mov c,a ; c is parent
    ;get first child of parent
    call goefch
    ana a
    jz remps ; parent has no child
    cmp b
    jz remoif ; obj is first child
    ;obj is not first child of parent
    ;find obj in the chain of children
    ; of its parent and remove it
remlp:
    mov c,a ; current child in c
    ;get next child in chain
    call goesib ; next child in a
    cmp b
    jz remfnd ; c's next sib is obj
    ana a ; 0 marks end of chain
    jz remps ; obj not found
    jmp remlp
remfnd:
    ;obj (b) is next sib of c
    ;remove obj from the chain
    ;next sib of obj now c's next sib
    mov a,b
    call goesib
    mov b,c
    call poesib
    jmp remps
remoif:
    ;obj (b) is 1st child of parent (c)
    ;make next sib of obj the 1st
    ; child of parent
    mov a,b
    call goesib
    mov b,c
    call poefch
remps:
    ;remove parent, sibling of obj
    lda op1
    mov b,a ; obj in b
    xra a
    call poepar
    xra a
    call poesib
    ret
    

;takes care of the insert_obj operation
;inpars: (op1 is object, op2 is parent)
;outpar: none
;DESTROYS ALL REGS
insobj:
    ;remove from old parent
    call remobj ;remove op1
    lda op1
    ;set object's new parent
    mov b,a ;object
    lda op2 ;parent
    call poepar
    ;object's new sibling is parent's
    ; old first child
    ; (obj is still in b)
    lda op2
    call goefch
    call poesib
    ;parent's new first child is obj
    ; (obj is still in b)
    lda op2
    mov c,b ;obj in c
    mov b,a ;parent in b
    mov a,c ;obj in a
    call poefch
    ret
    

;takes care of the get_prop_addr
; operation
;note that there is no property 0
;inpars: obj in a, property num in b
;outpar: property size byte in a
;        (0 if prop is 0 or not found)
;        address of prop in 0hl
;        (0 if prop is 0 or nf)
getpropadx:
    push b
    push d
    ana a
    jz gprpnf
    mov c,a ; obj in c
    call gopta
    call zgtblw ;text-length byte
    xchg ; prop tbl adx in de
    mov l,a
    mvi h,0
    dad h
    inx h
    dad d
    ;hl is now address of size byte
    ; of first property
    ;loop, advancing through prop.
    ; table looking for the prop.
gprplp:
    call zgtblw ; get size byte
    ana a ; zero marks end of props
    jz gprpnf
    mov d,a ;size byte
    ani 1fh ; lower 5 bits is prop num
    cmp b
    jz gprpf ;found prop
    jc gprpnf ;props are in descending
              ; order
    mov a,d ;size byte
    ;skip to next property
    ;upper 3 bits is prop length - 1
    rlc
    rlc
    rlc
    ani 7h
    inr a
    inr a ;+1 for size byte
    call aahl
    jmp gprplp
gprpf: ;found it
    inx h ; advance past size byte
    mov a,d
    pop d
    pop b
    ret
gprpnf: ;not found, or obj=0
    lxi h,0
    xra a
    pop d
    pop b
    ret


;handles a z-machine routine call
;stack frame was documented earlier
;inpars: none
;outpar: none
;MODIFIES ALL REGS
setuproutine:
    ;
    ; write #words used by caller into
    ;  caller's stack frame
    lhld zsp
    xchg
    lhld zlp
    call sub16
    mov b,h
    mov c,l
    lhld zlp
    dcx h
    call writezstack
    ;
    ; start prep of CALL INFO MSword
    ; note we will modify this below
    call rbizpc ;var for return val
    sta (tempcallinfom + 1)
    lda (zpc + 2) ;MSbit of zpc
    sta tempcallinfom
    ; save current zsp,
    ; push junk on zstk - this will be
    ;  replaced by CALL INFO MSword
    ;  below
    lhld zsp
    push h
    call zpush
    ;
    lhld zpc   ; LSword of caller's zpc
    call zpush ;  is CALL INFO LSword
    ;
    ; push 0 on zstk - this is a
    ;  placeholder for the number of
    ;  words used in this stack frame,
    ;  and is only made valid when this
    ;  routine calls another
    ; note 0 marks the top frame
    lxi h,0
    call zpush
    ;
    ; want new zlp where zsp is now
    lhld zsp
    xchg ; new zlp in de until
    ;      we save old zlp on stack
    ;
    ; save caller's zlp on zstk
    lhld zlp    ; caller's zlp
    call zpush
    ;
    ; now can write new zlp
    xchg
    shld zlp
    ;
    ; now zsp pts to start of locals
    ;  on zstack - save this
    lhld zsp
    push h
    ;
    ; put location of new routine
    ; (byte address) in cde
    mvi c,0
    lhld op1 ; location / 2
    mov d,h
    mov e,l
    call ahl24
    ; now cde = 2*op1 = location
    ;
    ; get num operands given by the
    ;  call opcode, not including
    ;  op1 which is routine location
    lda opcount
    dcr a
    mov l,a ; #operands in l
    ;
    ; 1st byte is #locals for routine
    call zgetb
    ani 0fh
    mov h,a   ; #locals in h
    call a124
    ;
    ; the operands to the call opcode
    ;  overwrite the locals, so we
    ;  can't copy more of them than
    ;  the #locals
    mov a,h
    sub l   ;#locals - #operands
    jnc opcok
    mov l,h
opcok:
    ;
    ; save #locals and #operands in
    ;  CALL INFO MSword
    ; #locals in bits 4..1
    ; #operands in bits 7..6
    push h
    mov a,h ;#locals
    add a   ; in bits 4..1
    mov h,a
    mov a,l ;#operands
    rrc
    rrc     ; in bits 7..6
    ora h
    lxi h,tempcallinfom
    ora m   ;MSbit of zpc
    mov m,a
    pop h
    ;
    ; next in routine location is
    ;  init values for the locals
    ; put them on zstk
    push h  ; save #locals, #operands
    mov a,h ; #locals
    inr a
    push psw
lclcp:
    pop psw
    dcr a
    jz elclcp
    push psw
    call zgetw
    call a224
    call zpush
    jmp lclcp
elclcp:
    ;
    ; cde now points to first opcode
    ;  of the routine
    ; save to zpc
    lxi h,zpc
    call putl
    ;
    pop h
    mov b,l ; #operands in b
    ;
    ; now zsp is where it should be
    ;  save it on stack because we're
    ;  about to use it again
    lhld zsp
    xthl ; zsp on stack,
    ;      start of locals in hl
    shld zsp
    ;
    ; now overwrite the default values
    ;  of the locals with as many
    ;  operands as we got with the
    ;  call opcode
    lxi h,op2
    inr b
lclow:
    dcr b
    jz elclow
    mov e,m
    inx h
    mov d,m
    inx h
    xchg
    call zpush
    xchg
    jmp lclow
elclow:
    ;
    ; restore correct zsp
    pop h
    shld zsp
    ;
    ; write CALL INFO MSword
    lhld tempcallinfom
    mov b,h
    mov c,l
    pop h
    call writezstack
    ret

;handles the return from a routine
; i.e. discards top stack frame
;stack frame was documented earlier
;inpars: none
;outpar: none
teardownroutine:
    push b
    push d
    push h
    lhld zlp
    dcx h
    lxi b,0
    ;WORDS USED = 0 marks top frame
    call writezstack
    dcx h
    push h
    call readzstack ; LSword CALL INFO
    xchg       ;  in de
    pop h
    dcx h
    shld zsp   ; caller's zsp restored
    call readzstack ; MSword CALL INFO
    mov a,l
    ani 01h
    mov c,a; zpc MSbit in c
    push h ;save CALL INFO MSword
    lxi h,zpc
    call putl  ; caller's zpc restored
    lhld zlp
    call readzstack
    shld zlp   ; caller's zlp restored
    pop h
    mov a,h; dst var # for return value
    pop h
    pop d
    pop b
    ret

;performs the branch at pc in mem, when
; condition of the operation was true
;if a is !0 on return from here,
; a return from the current z-machine
; routine should be performed, 
; returning the value in hl returned
; from here.
;inpars: none
;outpar: value to return (if any) in hl
;  bool in a indicating whether to ret
;PRESERVES flags (but not a)
branchtrue:
    mvi a,80h
    jmp branch
;
;same as above, but for case where
; condition of the operation was false
branchfalse:
    mvi a,0
;
;helper for branchtrue, branchfalse
branch:
    push b
    push d
    push psw
    mov c,a ; save condition
    ;
    call rbizpc ; first byte from zpc
    mov b,a ; save in b
    ani 40h
    mov a,b
    jz off14
    ;
    ; 6 LSbits are offset, unsigned
    ani 3fh
    mov l,a
    mvi h,0
    jmp offdon
    ;
off14:
    ; 6 LSbits of a are 6 MSbits of
    ;  offset, and all bits of next
    ;  byte at zpc are 8 LSbits of
    ;  offset, which is signed
    ani 3fh     ; 6 MSbits of offset
    mov h,a
    ; extend sign
    ani 20h
    jz sgndon
    mov a,h
    ori 0c0h
    mov h,a
sgndon:
    call rbizpc ; 8 LSbits of offset
    mov l,a
    ;
offdon:
    ; offset is now in hl
    ; condition and MSbit of the first
    ;  byte determine whether or not
    ;  to branch
    mov a,b
    xra c
    add a
    jnc coff0
    ;
    ; cond != MSbit so no branch no ret
    xra a  ; no return from z-routine
    jmp retbr
    ;
coff0:
    ; cond == MSbit ...
    mov a,h
    ora l
    jnz coff1
    ; ... AND offset == 0
    ;
    ; return 0 from z-routine
    mvi a,1
    jmp retbr
    ;
coff1:
    ; cond == MSbit ...
    mov a,h
    ana a
    jnz coffn
    mov a,l
    dcr a
    jnz coffn
    ; ... AND offset == 1
    ;
    ; return 1 from z-routine
    mvi a,1
    jmp retbr
    ;
coffn:
    ; cond == MSbit
    ; AND
    ; offset != 0
    ; AND
    ; offset != 1
    ;
    ; do the branch:
    ;  zpc = zpc + offset - 2
    dcx h
    dcx h
    push h ; save (offset - 2)
    lxi h,zpc
    call getl ; zpc in cde
    pop h     ; (offset - 2) in hl
    call ashl24
    lxi h,zpc
    call putl
    xra a  ; no return from z-routine
    ;
retbr:
    mov d,a
    pop psw ; restore flags
    mov a,d
    pop d
    pop b
    ret


    
;prints the encoded z-machine string at
; the given offset in zmem
;returns the z-mem address immediately
; following the string.
;inpars:  cde is 32-bit z-mem address
;          of start of string
;outpars: cde the 32-bit z-mem address
;          following the string
;modifies b
pzstr:
    push h
    ;
    ;set up for unpack (see below)
    lxi h,stradx
    call putl ; cur pos in string
    lxi h,0
    shld packed3 ; most-recently read wd
    xra a
    sta packed3count ;#zchars left in packed3
    ;
    ;loop, unpacking/decoding and
    ; outputting string characters
pzslp:
    call unpack
    cpi 6
    jc pzspec
    ;(a) is offset into alphabet 0
    lxi h,za0
pzalph:
    call aahl
    mov a,m
    call pzscii
    jmp pzslp
pzspec: ;not an alphabet 0 character
    ; (a) < 6 here
    ana a
    jz pspace ; zero printed as space
    cpi 4
    jc pzsabr ; print abbreviation
    ;next ch from:
    ; a=4: alphabet 1
    ; a=5: alphabet 2
    jnz pzsza2
pzsza1:
    ;next ch from alphabet 1
    call unpack
    cpi 6
    jc pzspec
    ;(a) is offset into alphabet 1
    lxi h,za1
    jmp pzalph
pzsza2:
    ;next ch from alphabet 2
    call unpack
    cpi 6
    jc pzspec
    jz p10bit ; char in next 2 zchars
    ;(a) is offset into alphabet 2
    lxi h,za2
    jmp pzalph
pspace:
    mvi a,' '
    call pzscii
    jmp pzslp
pzsabr:
    ;find entry in abbreviations table
    ;(a) is z
    ;next char is x
    ;we want 2* word at z-mem address
    ; zabbrevsbase + 2*(32*(z-1) + x)
    dcr a ; (z-1)
    mov l,a
    mvi h,0
    dad h
    dad h
    dad h
    dad h
    dad h ;hl = 32*(z-1)
    call unpack ;(a) is x
    call aahl ;hl = 32*(z-1) + x
    dad h ;hl = 2*(32*(z-1) + x)
    xchg
    lhld zabbrevsbase
    dad d ;hl = zabbrevsbase + 2*(32*(z-1)+x)
    call zgtwlw
    mov d,h
    mov e,l
    mvi c,0
    call ahl24 ;cde is adx of abbrev.
    ;print the abbreviation
    call pzabbr
    jmp pzslp
p10bit:
    ;next 2 zchars are a 10-bit value
    ; whose 8 LSbits are the zscii
    ; character to print
    call unpack
    rrc
    rrc
    rrc
    ani 0e0h
    mov b,a
    call unpack
    ani 1fh
    ora b
    call pzscii
    jmp pzslp
    ;
pzsret:
    ;unpack JUMPS here when string
    ; has no more characters
    ;return adx following str in cde
    lxi h,stradx
    call getl
    pop h
    ret
;
;helper for pzstr
;unpacks zchars from the encoded string
;returns the zchar read in a
;doesn't return if there are no more
unpack:
    push b
    push h
    lhld packed3
    lda packed3count
    ana a
    jnz unpnz ;unused zchars in hl
    ;there are no unused zchars in hl
    ;iff MSbit of hl is set, we have
    ; read the entire string
    ;otherwise, get the next packed wd
    mov a,h
    add a
    jnc unpnxt
    ;the entire string has been read
    ;we now return from pzstr with the
    ; address following the string in
    ; cde
    pop h
    pop b
    xthl
    pop h ; discard unpack ret adx
    jmp pzsret
unpnxt:
    ;get next packed word from string
    push d
    lxi h,stradx
    call getl
    call zgetw ;next packed wd in hl
    shld packed3
    call a224  ;address += 2
    lxi h,stradx
    call putl
    pop d
unp1st:
    ;unpack the first zchar into a
    lhld packed3
    mvi a,2
    sta packed3count ;2 zchars left in packed3
    mov a,h
    rrc
    rrc
    ani 1fh
    pop h
    pop b
    ret
unpnz:
    ; there are (a) unused zchars in hl
    dcr a
    sta packed3count ;this many will be left
    jz unp3rd
unp2nd:
    ;unpack the second zchar into a
    mov a,h
    rlc
    rlc
    rlc
    ani 18h
    mov b,a
    mov a,l
    rlc
    rlc
    rlc
    ani 7h
    ora b
    pop h
    pop b
    ret
unp3rd:
    ;unpack the third zchar into a
    mov a,l
    ani 1fh
    pop h
    pop b
    ret
;
;helper for pzstr
;prints abbreviation at address cde in
; zmem
;MODIFIES all registers
pzabbr:
    lda inabr
    ana a
    cnz error ;can't nest abbreviations
    mvi a,1
    sta inabr
    ;
    ;save current stradx, packed3, packed3count
    lhld stradx
    push h
    lda (stradx+2)
    push psw
    lhld packed3
    push h
    lda packed3count
    push psw
    ;
    ;print the abbreviation string
    call pzstr
    ;
    ;restore stradx, packed3, packed3count
    pop psw
    sta packed3count
    pop h
    shld packed3
    pop psw
    sta (stradx+2)
    pop h
    shld stradx
    xra a
    ;not in abbreviation any more
    sta inabr
    ;
    ret


;prints (signed 16-bit) number
;inpar: (number in op1)
;outpar: none
;MODIFIES ALL REGS
printnum:
    lhld op1
    ;if negative, print '-' and do abs
    mov a,h
    add a
    jnc pnmnn
    mvi a,'-'
    call pzscii
    call negate
pnmnn:
    ;if number is zero, print '0'
    mov a,h
    ora l
    jnz pnmnz
    mvi a,'0'
    call pzscii
    ret
pnmnz: ;print positive number
    xchg ; number is in de
    mvi b,0 ; counts digits
pnmlp:
    lxi h,10
    call div16u; de quotient, hl rem.
    mvi a,'0'
    add l ; a = ascii of digit
    push psw ; save it on stack
    inr b
    mov a,d
    ora e
    jnz pnmlp
    ;now the digits are on the stack
    ;print them
pnmplp:
    pop psw
    call pzscii
    dcr b
    jnz pnmplp
    ret
    
;outputs a ZSCII character, or '?' if
; not a legal character
;NOTE currently doesn't support output
; characters other than standard ascii
; (32 to 126) and 0 which prints
; nothing
;inpar: the character in a
;outpar: none
pzscii:
    ana a
    rz ; zero is okay - prints nothing
    cpi 13      ;newline
    jz ptoscr
    ; 32 to 126 are standard ascii
    cpi 32
    jc pillgl
    cpi 127
    jnc pillgl
    ; standard ascii
    jmp ptoscr
    ;
pillgl:
    ;illegal for output - print '?'
    mvi a,'?'
ptoscr:
    call printchar
    ret


;gets up to 6 chars of text from
; keyboard to filnam, echoing to
; display
;only allows letters and digits
;only allows letters for first char
;converts letters to uppercase
;pads to 6 chars with spaces
getfilename:
    push b
    push d
    push h
    lxi h,filnam
    mvi b,6 ;b = max # chars to write
    mvi c,0 ; wrote this many so far
gfnllp:
    call safechget; get char from keyboard
    cpi 0dh
    jz gfnnewl ; newline
    cpi 1dh
    jz gfnbacksp ; backspace
    cpi 08h
    jz gfnbacksp ; backspace
    cpi 7fh
    jz gfnbacksp ; backspace
    ;prevent buffer overflow
    mov d,a
    mov a,b
    cmp c
    jz gfnllp
    mov a,d
    ;check if it is legal for input
    dcr c
    inr c
    jz gfnlettonly
    cpi '0'
    jc gfnllp
    cpi ('9' + 1)
    jc gfnlgl
gfnlettonly:
    cpi 'A'
    jc gfnllp
    cpi ('Z' + 1)
    jc gfnlgl
    cpi 'a'
    jc gfnllp
    cpi ('z' + 1)
    jnc gfnllp
    ;convert lowercase to uppercase
    adi ('A' - 'a')
gfnlgl:
    ;echo to screen
    call safelcd
    ;write to text buf
    mov m,a
    inx h
    inr c
    jmp gfnllp
gfnbacksp:
    ;can't go back if haven't typed
    ; anything yet
    mov a,c
    ana a
    jz gfnllp
    ;do backspace
    mvi a,08h ; back one position
    call safelcd
    mvi a,' '
    call safelcd
    mvi a,08h ; back one position
    call safelcd
    dcx h
    dcr c
    jmp gfnllp
gfnnewl:
    ; echo to screen
    mvi a,0dh ; carriage return
    call safelcd
    mvi a,0ah ; newline
    call safelcd
    ;pad with ' ' to 6 chars
gfnpad:
    inr c
    mov a,c
    cpi 7
    jz rgfnline
    mvi m,' '
    inx h
    jmp gfnpad
rgfnline:
    pop h
    pop d
    pop b
    ret

;gets text from keyboard, echoing to
; display
;NOTE currently doesn't support input
; characters other than standard ascii
; (32 to 126) -> only these will be
; written to text buf
;inpar: hl is z-mem address of text buf
;        format as given for sread op
;outpar: (modified text buf)
;PRESERVES hl
getline:
    push b
    push d
    push h  ;points to text buf
    call zgtblw ;1st byte has size
    inx h   ;start writing text here
    ana a
    jz rgtline
    dcr a
    mov b,a ;b = max # chars to write
    mvi c,0 ; wrote this many so far
gtllp:
    call safechget; get char from keyboard
    cpi 0dh
    jz gtnewl ; newline
    cpi 1dh
    jz backsp ; backspace
    cpi 08h
    jz backsp ; backspace
    cpi 7fh
    jz backsp ; backspace
    ;prevent buffer overflow
    mov d,a
    mov a,b
    cmp c
    jz gtllp
    mov a,d
    ;check if it is legal for input
    cpi 32
    jc gtllp   ; < 32 not a legal char
    cpi 127
    jnc gtllp  ; > 126 not a legal char
    ;echo to screen
    call safelcd
    ;convert to lowercase
    cpi 'A'
    jc gtwb
    cpi ('Z'+1)
    jnc gtwb
    adi ('a'-'A')
gtwb:
    ;write to text buf
    call zptblw
    inx h
    inr c
    jmp gtllp
backsp:
    ;can't go back if haven't typed
    ; anything yet
    mov a,c
    ana a
    jz gtllp
    ;do backspace
    mvi a,08h ; back one position
    call safelcd
    mvi a,' '
    call safelcd
    mvi a,08h ; back one position
    call safelcd
    dcx h
    dcr c
    jmp gtllp
gtnewl:
    ; echo to screen
    mvi a,0dh ; carriage return
    call safelcd
    mvi a,0ah ; newline
    call safelcd
rgtline:
    mvi a,0 ; 0 terminates line
    call zptblw
    pop h
    pop d
    pop b
    ret


;cleanup after screen output
; (when game ending)
;inpar: none
;outpar: none
sccln:
    ;advance one line
    mvi a,13
    call safelcd
    mvi a,10
    call safelcd
    ret


;*** the [more] function ***
;When outputting large amounts of text
; it is possible that not all of it can
; fit on the screen.  To prevent text
; from scrolling off the screen before
; the reader can read it, we output it
; in chunks that do fit the screen,
; asking that the reader press a key to
; display each subsequent screenful
; until the end of the text.  To show
; the reader that there is more text to
; display, a "[more]" prompt is shown
; on the bottom line after each
; screenful of text.
;The morebegin, moreend and printchar routines
; below implement the [more] function.

;indicates that a block of [more]-
; function text is about to start
;inpar: none
;outpar: none
morebegin:
    push h
    xra a
    sta linecount
    sta lbcount
    sta lbspacecount
    lxi h,linebuf
    shld lbnextptr
    pop h
    ret

;indicates end of output of a [more]-
; block of text
;inpar: none
;outpar: none
moreend:
    push h
    push d
    ;write whatever is in linebuf out
    ; to the screen
    lhld lbnextptr
    lxi d,linebuf
    xchg
    call sub16
    mov a,l ; this many to screen
    lxi h,linebuf
    call printstring
    ;reset line buffering vars
    call morebegin
    pop d
    pop h
    ret
    
morestr: db '[more]'
morestrend:

;prints the character in a, handles
; [more] functionality
;inpar: character in a
;outpar: none
printchar:
    push b
    push d
    push h
    mov b,a ; b is char to print
    cpi 0dh
    jz pbuf
    cpi ' '
    lda lbcount
    jnz notspc
    sta lbspacecount ;#chars before last ' '
notspc:
    inr a
    sta lbcount
    mov c,a ; c is lbcount
    lhld lbnextptr
    mov m,b ; write char to linebuf
    inx h
    shld lbnextptr
    mov a,c
    cpi 41
    jz pbuf ; linebuf full - print
    pop h
    pop d
    pop b
    ret
pbuf: ;end of line - print to screen
    lda lbcount
    mov b,a ; b is lbcount
    cpi 41
    jnz pachrs; print all in linebuf
    ;printing because linebuf is full
    ; if the last space is within 16
    ; positions from the end of linebuf,
    ; print up to the space and move
    ; everything after the space to the
    ; start of linebuf to go out with
    ; the next line
    lda lbspacecount
    mov c,a ; c is lbspacecount
    adi 16
    sub b
    mvi a,40
    jc pachrs ; print 40 chars
    ;print up to last ' '
    mov a,c
pachrs:
    ;b is lbcount
    ;print (a) chars from linebuf, move
    ; rest of linebuf to start of linebuf
    ; to go out with next line
    lxi h,linebuf
    mov d,h
    mov e,l ; linebuf in de for pnocpy
    mov c,a ; c is # chars printed
    call printstring ; print (a) chars
    ;if printed < 40chars, advance line
    ; (advances automatically if 40)
    mov a,c
    cpi 40
    jz pnoadv
    mvi a,0dh
    call safelcd
    mvi a,0ah
    call safelcd
pnoadv:
    ;increment linecount and if 7 lines
    ; have been written, need to prompt
    ; and wait for key
    lda linecount
    inr a
    cpi 7
    jnz pnomor
    ;print prompt
    push h
    mvi a,(morestrend-morestr)
    lxi h,morestr
    call printstring
    pop h
    ;wait for key press
    call safechget
    ;erase prompt
    mvi a,(morestrend-morestr)
    call erascr
    xra a
pnomor:
    sta linecount
    ;need to copy remaining unprinted
    ; to start of linebuf
    xra a
    sta lbspacecount
    mov a,b ;lbcount
    sub c   ;- #chars printed
    mov c,a ;c = # chars left in linebuf
    jz pnocpy
    ;don't usually want leading spaces
    ; at start of a line, so skip past
    ; up to 2 spaces here (since often
    ; sentences are ended with 2 ' 's).
    ;if next char is ' ', don't copy it
    mov a,m
    cpi ' '
    jnz nspsk
    ;skip the space
    inx h
    dcr c
    jz pnocpy
nspsk:
    mov a,m
    cpi ' '
    jnz nspsk2
    ;skip the space
    inx h
    dcr c
    jz pnocpy
nspsk2:
    ;do the copy
    ;c is #chars to copy
    mov b,c
    ;linebuf still in de
pcpylp:
    mov a,m
    inx h
    stax d
    inx d
    dcr b
    jnz pcpylp
pnocpy:
    ;de -> 1st unused spot in linebuf
    xchg
    shld lbnextptr
    mov a,c
    sta lbcount
    pop h
    pop d
    pop b
    ret

;backspaces over given number of chars
; on screen
;inpar: a is # chars
;outpar: none
erascr:
    ana a
    rz
    push b
    mov b,a
    mov c,b
    mvi a,8 ; move cursor back
eralp1:
    call safelcd
    dcr c
    jnz eralp1
    mov c,b
    mvi a,' '
eralp2:
    call safelcd
    dcr c
    jnz eralp2
    mvi a,8 ; move cursor back
eralp3:
    call safelcd
    dcr b
    jnz eralp3
    pop b
    ret

;puts given # bytes from string to
; screen through safelcd
;inpars: hl is adx of string
;        a is #bytes
;outpar: hl points one past last byte
printstring:
    ana a
    rz
    push b
    mov b,a
psslp:
    mov a,m
    call safelcd
    inx h
    dcr b
    jnz psslp
    pop b
    ret
    

;compares de and hl as signed ints
;(I want cy=1 to indicate that
; signed hl > signed de)
;inpars: signed ints in de and hl
;outpars: flags
;PRESERVES all regs except a
cmp16s:
    push b
    ;since we are treating these as
    ; signed, need to invert MSbits
    ; so cy will be set correctly
    mov a,d
    xri 80h
    mov b,a
    mov a,h
    xri 80h
    mov c,a
    ;cmp be - cl
    mov a,b
    sub c
    jnz c16srt
    mov a,e
    sub l
c16srt:
    pop b
    ret

;as above, but unsigned
cmp16:
    mov a,d
    sub h
    rnz
    mov a,e
    sub l
    ret

;subtracts hl from de
;inpars: ints in de and hl
;outpar: result in hl
;NOTE flags indicate unsigned subtract
sub16:
    mov a,e
    sub l
    mov l,a
    mov a,d
    sbb h
    mov h,a
    ret

;multiplies de and hl as signed 16-bit
; integers
;inpars: the signed ints in de and hl
;outpar: signed 16-bit result in hl
mul16s:
    push b
    push d
    ; determine sign for result
    mov a,h
    xra d
    add a
    push psw
    ; take absolute values
    mov a,h
    add a
    cc negate
    xchg
    mov a,h
    add a
    cc negate
    ; move hl to bc
    mov b,h
    mov c,l
    ; do unsigned multiply bc * de
    call mul16u ; result in hl
    ; fix sign
    pop psw
    cc negate
    pop d
    pop b
    ret

;func to multiply an unsigned 8-bit
; number by an unsigned 16-bit number
;inpars: numbers to multiply in bc, d
;outpar: result in dehl
mul816u:
    mvi a,8
    jmp mulu

;func to multiply two 16-bit unsigned
; numbers returning a 32-bit unsigned
; result
;inpars: the two numbers to multiply
; in bc and de
;outpar: result in dehl
mul16u:
    mvi a,16
mulu:
    push b
    push d ; will do bc * (*sp)
    lxi d,0
    lxi h,0 ; dehl=0000h
mulbit:
    ; check if done ;
    dcr a
    jm mulret
    ; shift dehl up ;
    push psw
    dad h
    mov a,e
    ral
    mov e,a
    mov a,d
    ral
    mov d,a
    pop psw
    ; shift *sp up and, iff msb was set ;
    ; add bc into dehl ;
    xthl
    dad h
    xthl
    jnc mulbit
    dad b ; hl = hl + bc, maybe cy
    jnc mulbit
    inx d ; de = de + 1 (since cy=1)
    jmp mulbit
mulret:
    pop b ; leftover junk word
    pop b ; restore original bc
    ret

;divides de by hl as signed 16-bit
; integers, returns quotient and rem.
;inpars: the signed ints in de and hl
;outpar: signed 16-bit quotient in de
;        signed 16-bit mod in hl
div16s:
    push b
    ; check for div by 0
    mov a,h
    ora l
    cz error
    ; determine signs for results
    mov a,d
    add a
    push psw ; sign for mod
             ;  (same as de)
    mov a,h
    xra d
    add a
    push psw ; sign for quotient
    ; take absolute values
    mov a,h
    add a
    cc negate
    xchg
    mov a,h
    add a
    cc negate
    xchg
    ; do unsigned divide
    call div16u
    ; quotient in de, mod in hl
    ; fix signs
    pop psw ; sign for quotient
    xchg
    cc negate
    pop psw ; sign for mod
    xchg
    cc negate
    pop b
    ret

;func to divide two 16-bit unsigned
; numbers returning 16-bit unsigned
; quotient and remainder.
;inpars: the two numbers in de and hl,
; where the operation is (de)/(hl)
;0 in hl results in garbage
;outpars: quotient in de, remainder in
; hl
div16u:
    push b
    ; we're doing F/G where currently ;
    ; F is in de and G is in hl ;
    xchg
    mov b,d
    mov c,e
    lxi d,1
    push d
    dcr e
    ; now bcde contains G << 16       ;
    ; hl contains F, and *sp is 1     ;
    ; in each iteration of the loop   ;
    ; below *sp will be shifted left  ;
    ; and when the 1 shifts out the   ;
    ; top we know we're done -- also  ;
    ; the next quotient bit shifts    ;
    ; into *sp from the right ;
    ana a ; cy = 0
divloop:
    ; shift G down 1 bit ;
    ; (cy = 0 here always)
    mov a,b
    rar
    mov b,a
    mov a,c
    rar
    mov c,a
    mov a,d
    rar
    mov d,a
    mov a,e
    rar
    mov e,a
    ; save F ;
    push h
    ; subtract shifted G from F ;
    mov a,l
    sub e
    mov l,a
    mov a,h
    sbb d
    mov h,a
    mvi a,0
    sbb c
    mvi a,0
    sbb b
    ; if underflowed, quotient bit is ;
    ; 0 and we must restore old F     ;
    ; else quotient bit is 1, leave F ;
    cmc ; cy = quotient bit
    jc noborrow
    ; underflowed ;
    pop h
    jmp shiftquot
noborrow:
    ; didn't underflow ;
    inx sp
    inx sp
shiftquot:
    ; shift quotient bit into *sp ;
    xthl
    mov a,l
    ral
    mov l,a
    mov a,h
    ral
    mov h,a
    xthl
    ; if 1 shifted out, we're done ;
    jnc divloop
    ; now we have the finished ;
    ; quotient in *sp, remainder in hl ;
    pop d
    pop b
    ret

;negates a 16-bit signed integer
;inpar: 16-bit signed integer in hl
;outpar: 16-bit signed integer in hl
negate:
    ; complement and increment
    mov a,h
    cma
    mov h,a
    mov a,l
    cma
    mov l,a
    inx h
    ret

;tests a bit in a 32-bit word
;inpars: 32-bit word in bcde
;        bit number in a, counting
;         starting from the left
;         i.e. leftmost is bit 0
;         (must be 0..31)
;outpar: z flag
testbit:
    push h
    call bitpos
    dcr l
    jz tstb
    dcr l
    jz tstc
    dcr l
    jz tstd
tste:
    ana e
    pop h
    ret
tstb:
    ana b
    pop h
    ret
tstc:
    ana c
    pop h
    ret
tstd:
    ana d
    pop h
    ret
    
;sets a bit in a 32-bit word
;inpars: 32-bit word in bcde
;        bit number in a, counting
;         starting from the left
;         i.e. leftmost is bit 0
;         (must be 0..31)
;outpar: modified bcde
setbit:
    push h
    call bitpos
    dcr l
    jz setb
    dcr l
    jz setc
    dcr l
    jz setd
sete:
    ora e
    mov e,a
    pop h
    ret
setb:
    ora b
    mov b,a
    pop h
    ret
setc:
    ora c
    mov c,a
    pop h
    ret
setd:
    ora d
    mov d,a
    pop h
    ret
    
;clears a bit in a 32-bit word
;inpars: 32-bit word in bcde
;        bit number in a, counting
;         starting from the left
;         i.e. leftmost is bit 0
;         (must be 0..31)
;outpar: modified bcde
clearbit:
    push h
    call bitpos
    cma
    dcr l
    jz clrb
    dcr l
    jz clrc
    dcr l
    jz clrd
clre:
    ana e
    mov e,a
    pop h
    ret
clrb:
    ana b
    mov b,a
    pop h
    ret
clrc:
    ana c
    mov c,a
    pop h
    ret
clrd:
    ana d
    mov d,a
    pop h
    ret

bittable: db 80h, 40h, 20h, 10h
        db 08h, 04h, 02h, 01h
;helper for above
;gets which byte of bcde and which
; bit of the byte is to be modified
;inpar: bit number in a, 0..31
;outpars:bit mask in a, 
;        l=1,2,3,4 means bit in b,c,d,e
bitpos:
    ; l=1,2,3,4 means bit in b,c,d,e
    mvi l,0
bytlp:
    inr l
    sui 8
    jnc bytlp
    ;now l indicates which byte,
    ; and a+8 is bit position in byte
    adi 8
    push h
    lxi h,bittable
    ; add bit position to bittable
    call aahl
    mov a,m ;read pattern from bittable
    pop h
    ret
    

;gets object's attribute flags
;inpar: object in h
;outpar: flags in bcde
goeaf:
    push h
    mov b,h
    mvi c,0
    call goeadx
    call zgtllw
    pop h
    ret

;puts object's attribute flags
;inpar: object in h, flags in bcde
;outpar: none
poeaf:
    push h
    push b
    mov b,h
    mvi c,0
    call goeadx
    pop b
    call zptllw
    pop h
    ret

;gets object's parent
;inpar: object in a
;outpar: parent in a
goepar:
    push b
    mov b,a
    mvi c,4
    call goeb
    pop b
    ret
    
;puts object's parent
;inpar: object in b, new parent in a
;outpar: none
poepar:
    push b
    mvi c,4
    call poeb
    pop b
    ret
    
;gets object's next sibling
;inpar: object in a
;outpar: parent in a
goesib:
    push b
    mov b,a
    mvi c,5
    call goeb
    pop b
    ret
    
;puts object's next sibling
;inpar: object in b, new sib in a
;outpar: none
poesib:
    push b
    mvi c,5
    call poeb
    pop b
    ret
    
;gets object's first child
;inpar: object in a
;outpar: first child in a
goefch:
    push b
    mov b,a
    mvi c,6
    call goeb
    pop b
    ret
    
;puts object's first child
;inpar: object in b, new child in a
;outpar: none
poefch:
    push b
    mvi c,6
    call poeb
    pop b
    ret
    
;gets address of object's prop table
;inpar: object in a
;outpar: 0hl is zmem address
gopta:
    push b
    mov b,a
    mvi c,7
    call goeadx
    call zgtwlw
    pop b
    ret

;gets a byte from goeadx(obj) + offset
;inpars: obj in b, offset in c
;outpar: the byte read in a
goeb:
    push h
    call goeadx
    call zgtblw
    pop h
    ret
    
;puts a byte to goeadx(obj) + offset
;inpars: obj in b, offset in c,
;        new value in a
;outpar: none
poeb:
    push h
    push psw
    call goeadx
    pop psw
    call zptblw
    pop h
    ret

;gets the zmem address of the object
; table entry for the given object and
; returns that address plus an offset
;inpar: obj in b, offset in c
;outpar: hl is zmem address
goeadx:
    push b
    push d
    ; address of entry is
    ;  zobjectstable + 9*(object number)
    ; has to be in lower 64k so no
    ;  overflow will happen here
    mvi h,0
    mov l,b ;hl = (object number)
    dad h
    dad h
    dad h   ;hl = 8*(object number)
    mov a,b
    call aahl ;hl = 9*(object number)
    xchg
    lhld zobjectstable
    dad d
    ; now hl is address of obj entry
    mov a,c ; add offset
    call aahl
    pop d
    pop b
    ret


;gets the operand of the given type,
; from z-machine mem at zpc, and takes
; care of incr'g zpc past operand.
; Also increments opcount unless type is
; "none".
;inpar: type in a
;        must be 0, 1, 2 or 3
;outpar: operand in hl
getoperand: ;GetOperand
    cpi 3 ; type = "none"
    rz
    push b
    mov b,a
    lda opcount
    inr a
    sta opcount
    mov a,b
    pop b
    ana a
    jz getlargeconst    ;large constant
    dcr a
    jz getsmallconst    ;small constant
    jmp getvar  ;variable

;gets the small constant operand
; from z-machine mem at zpc, and takes
; care of incr'g zpc past operand.
;inpar: none
;outpar: the small const in hl
getsmallconst: ;GetSmallConstant
;;    call rbizpc
    call getinstbufbyte
    mov l,a
    mvi h,0
    ret

;gets the large constant operand
; from z-machine mem at zpc, and takes
; care of incr'g zpc past operand.
;inpar: none
;outpar: the large const in hl
getlargeconst:
;;    push b
;;    push d
;;    lxi h,zpc
;;    call getl
;;    call zgetw
;;    push h ;save the value
;;    ;increment zpc
;;    call a224
;;    lxi h,zpc
;;    call putl
;;    pop h
;;    pop d
;;    pop b
;;    ret
    push d
    call getinstbufword
    xchg
    pop d
    ret

;gets the variable operand given by the
; byte in z-machine mem at zpc, & takes
; care of incr'g zpc past operand.
;inpar: none
;outpar: the variable value in hl
getvar:
;;    call rbizpc
    call getinstbufbyte
    jmp getgivenvar
    
;gets the variable given by byte in a
;inpar: variable is given by a
;outpar: the variable value in hl
getgivenvar:
    ana a
    jnz ntos
    ; top of stack
    ;; ?? should I be popping this
    ;; or just reading it ??
    jmp zpop
ntos:
    cpi 16
    jnc getglobal ; global var
    ; local var
    jmp getlocal

;writes the value in hl to the variable
; given by the byte at zpc, and
; increments zpc
;inpar: value in hl
;outpar: none
putvar:
    call rbizpc
    jmp putgivenvar

;writes the value in hl to the variable
; given by a
;inpars: value in hl, variable in a
;outpar: none
putgivenvar:
    ana a
    jnz ntos2
    ; top of stack
    ;; ?? should I be pushing this
    ;; or just writing ??
    jmp zpush
ntos2:
    cpi 16
    jnc putglobal ;global var
    ; local var
    jmp putlocal


;gets the local variable given in a
;inpar: variable given by a
;       (must be 1..15)
;outpar: the variable value in hl
getlocal:
    lhld zlp
    call aahl ;hl = zlp+var
    call readzstack
    ret
    
;writes the value in hl to the local
; variable given by a
;inpars: value in hl,
;        variable given by a
;        (must be 1..15)
;outpar: none
putlocal:
    push b
    push h
    mov b,h
    mov c,l
    lhld zlp
    call aahl ; hl = zlp+var
    call writezstack
    pop h
    pop b
    ret

;gets the global variable given in a
;inpar: variable given by a
;  (must be 16..255)
;outpar: the variable value in hl
getglobal:
    push d
    call globaladx
    call zgtwlw
    pop d
    ret
    
;writes the value in hl to the global
; variable given by a
;inpars: value in hl,
;        variable given by a
;        (must be 16..255)
;outpar: none
putglobal:
    push d
    push h
    push h
    call globaladx
    pop d
    call zptwlw
    pop h
    pop d
    ret

;helper for getglobal, putglobal
;gets the byte address of the global
; variable given by a
;globals are in lower 64k of zmem
;inpar: variable given by a
;        (must be 16..255)
;outpar: address in 0hl
;MODIFIES de
globaladx:
    lhld zglobalsbase
    sui 16 ; a = var-16
    mvi d,0
    mov e,a
    dad d
    dad d ;0hl = zglobalsbase + 2*(var-16)
    ret


;reads 10 bytes from z-machine mem at
; zpc into instbuf, leaving zpc as-is
;sets a pointer to the next available
; byte in instbuf, and expects that
; bytes will be read from instbuf using
; getinstbufbyte, getinstbufword functions
;DESTROYS ALL REGS
readinst:
    lxi h,zpc
    call getl ; zpc in cde
    lxi h,instbuf
    shld instbufptr ;pts to next byte
    mvi a,10
    call zgeta
    ret

;accessor to get a byte from next loc
; in instbuf
;outpar: byte in a
getinstbufbyte:
    push h
    lhld instbufptr
    mov a,m
    inx h
    shld instbufptr
    pop h
    ret
    
;accessor to get 16-bit word from next
; loc in instbuf (big-endian)
;outpar: word in de
getinstbufword:
    push h
    lhld instbufptr
    mov d,m
    inx h
    mov e,m
    inx h
    shld instbufptr
    pop h
    ret
    
;once we are done with instbuf, this
; function increments zpc past all the
; bytes we used
;DESTROYS ALL REGS
fixzpc:
    lhld instbufptr
    xchg
    lxi h,instbuf
    call sub16
    push h ;# to add into zpc
    lxi h,zpc
    call getl ; zpc in cde
    pop h
    call ahl24
    lxi h,zpc
    call putl
    ret


;reads byte from z-machine mem at zpc
; and increments zpc
;inpar: none
;outpar: byte in a
rbizpc:
    push b
    push d
    push h
    lxi h,zpc
    call getl
    call zgetb
    push psw ; save a
    ;increment zpc
    call a124
    call putl
    pop psw ; a contains the value read
    pop h
    pop d
    pop b
    ret

;pushes the value in hl onto the
; z-machine stack
;inpar: value in hl
;outpar: none
zpush:
    push b
    push h
    mov b,h
    mov c,l
    lhld zsp
    call writezstack
    inx h
    shld zsp
    pop h
    pop b
    ret

;pops a word off the z-machine stack
;inpar: none
;outpar: popped value in hl
zpop:
    lhld zsp
    dcx h
    shld zsp
    jmp readzstack

;writes bc to zstk[hl]
;inpars: val in bc, index in hl
;outpar: none
writezstack:
    push h
    ;error-check index
    mvi a,0fch ;valid index in [0,3ffh]
    ana h
    cnz error
    ;do the write
    call getzstackadx
;;    mov m,c
    mov m,b
    inx h
;;    mov m,b
    mov m,c
    pop h
    ret
    
;reads zstk[hl] into hl
;inpars: index in hl
;outpar: val in hl
readzstack:
    push d
    ;error-check index
    mvi a,0fch ;valid index in [0,3ffh]
    ana h
    cnz error
    ;do the read
    call getzstackadx
;;    mov e,m
    mov d,m
    inx h
;;    mov d,m
    mov e,m
    xchg
    pop d
    ret

;gets zstk + 2*hl (computes zstk adx)
;inpar: hl
;outpar: result in hl
getzstackadx: ;GetZStackAddress
    push d
    xchg
    lhld zstk
    xchg
    dad h
    dad d
    pop d
    ret
    

;adds the value in a to the 16-bit
; int in hl
;inpar: 16-bit int in hl
;       value to add in a
;outpar: updated hl
aahl:
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    ret

;;;adds 1 to the 32-bit int in cde
;;;inpars: 32-bit int in bcde
;;;outpar: result in bcde
;adds 1 to the 24-bit int in cde
;inpars: 24-bit int in cde
;outpar: result in cde
;;a132:
a124:
    mvi a,1
;;    call aa32
    call aa24
    ret
    
;;;adds 2 to the 32-bit int in bcde
;;;inpars: 32-bit int in bcde
;;;outpar: result in bcde
;adds 2 to the 24-bit int in cde
;inpars: 24-bit int in cde
;outpar: result in cde
;;a232:
a224:
    mvi a,2
;;    call aa32
    call aa24
    ret
    

;;;adds the value in a to the 32-bit
;;; int in bcde
;;;inpars: 32-bit int in bcde
;;;        val to add in a
;;;outpar: result in bcde
;adds the value in a to the 24-bit
; int in cde
;inpars: 24-bit int in cde
;        val to add in a
;outpar: result in cde
;;aa32:
aa24:
    add e
    mov e,a
    mvi a,0
    adc d
    mov d,a
    mvi a,0
    adc c
    mov c,a
;;    mvi a,0
;;    adc b
;;    mov b,a
    ret

;;;adds the value in hl to the 32-bit
;;; int in bcde
;;;inpars: 32-bit int in bcde
;adds the value in hl to the 24-bit
; int in cde
;inpars: 24-bit int in cde
;        val to add in hl
;outpar: result in cde
;MODIFIES b
;PRESERVES hl
;;ahl32:
ahl24:
    mvi b,0
doadd:
    mov a,l
    add e
    mov e,a
    mov a,h
    adc d
    mov d,a
    mov a,b
    adc c
    mov c,a
    ret
;same as above, but signed
;MODIFIES b
ashl24:
    mov a,h
    add a
    mvi b,0     ;extended sign of hl
    jnc doadd
    mvi b,0ffh  ;extended sign of hl
    jmp doadd
    
    
;gets a byte from lower 64k of
; z-machine mem
;inpar: 0hl is zm address of byte
;outpar: the byte in a
zgtblw:
    push b
    push d
    push h
    mvi c,0
    xchg
    call zgetb
    pop h
    pop d
    pop b
    ret

;puts a byte to lower 64k of
; z-machine mem
;inpars: a is the value to put,
;        0hl is zm address of byte
;outpar: none
zptblw:
    push b
    push d
    push h
    sta temp32
    lxi d,temp32 ; de is src adx
    mvi a,1
    call zputa
    pop h
    pop d
    pop b
    ret

;gets a byte from z-machine mem
;inpar: cde is zm address of byte
;outpar: the byte in a
zgetb:
    push b
    push d
    push h
    lxi h,temp32
    mvi a,1
    call zgeta
    lda temp32
    pop h
    pop d
    pop b
    ret
    
;gets a 16-bit word from lower 64k
; of z-machine mem
;inpar: 0hl is zm address of word
;outpar: the word in hl
zgtwlw:
    push b
    push d
    mvi c,0
    xchg
    call zgetw
    pop d
    pop b
    ret

;puts a 16-bit word to lower 64k of
; z-machine mem
;inpars: de is the value to put,
;        0hl is zm address of word
;outpar: none
zptwlw:
    push b
    push d
    push h
    xchg
    mov a,h ; z-mem is big-endian
    mov h,l ; 80c85 is little-endian
    mov l,a
    shld temp32
    xchg         ; 0hl is dest adx
    lxi d,temp32 ; de is src adx
    mvi a,2
    call zputa
    pop h
    pop d
    pop b
    ret

;;;gets a 16-bit word from z-machine mem
;;;inpar: bcde is zm address of word
;;;outpar: the word in hl
;gets a 16-bit word from z-machine mem
;inpar: cde is zm address of word
;outpar: the word in hl
zgetw:
    push b
    push d
    lxi h,temp32
    mvi a,2
    call zgeta
    lhld temp32
    mov a,h ; z-mem is big-endian
    mov h,l ; 80c85 is little-endian
    mov l,a
    pop d
    pop b
    ret
    
;gets a 32-bit word from lowest 64k
; of z-machine mem
;inpar: 0hl is zm address of word
;outpar: the word in bcde
zgtllw:
    push h
    xchg
    mvi c,0 ;cde is zm address of word
    lxi h,temp32
    mvi a,4
    call zgeta
    lxi h,temp32
    mov b,m
    inx h
    mov c,m
    inx h
    mov d,m
    inx h
    mov e,m
    pop h
    ret
    
;puts a 32-bit word to lowest 64k of
; z-machine mem
;inpars: bcde is the value to put,
;        0hl is zm address of word
;outpar: none
zptllw:
    push b
    push d
    push h
    lxi h,temp32
    mov m,b
    inx h
    mov m,c
    inx h
    mov m,d
    inx h
    mov m,e
    lxi d,temp32 ; de=src
    pop h ; 00hl=dest in zmem
    push h
    mvi a,4
    call zputa
    pop h
    pop d
    pop b
    ret

;gets array of bytes from z-machine mem
;OPT ROM must be selected before this
; is called
;leaves OPT ROM selected
;leaves REX block rexblock[0] selected in
; OPT ROM space
;inpars:
;   a is the number of bytes (>=1)
;   cde adx of z-machine mem source
;    (17-bit)
;   hl pts to dest array in local mem
;outpar: none
;MAY DESTROY ALL REGS
zgeta:
    mov b,a ;#bytes saved in b
    ;check if src adx is in dyn mem
    ;note dyn mem is at most 20477 byte
    push h
    mov a,c
    ana a
    jnz gtup64 ; in upper 64k
    lhld mdynsize
    dad d
    jc gtlw64 ; in lower 64k not dyn
    ;adx is in dynamic zmem
    ;read from ram
    lhld dynbase
    dad d
    xchg  ;de src adx
    pop h ;hl dst adx
zgtlp2:
    ldax d
    mov m,a
    dcr b
    rz
    inx d
    inx h
    jmp zgtlp2
gtup64:
    ;read from upper 64k of game rom
    mvi h,2 ;rexblock[h] REX block num
    mov a,d
    ;in block 2, offset of start of
    ; upper 64k of game data=33h*256,
    ; and there are 4ch*256 bytes of
    ; upper-64k game data
    mvi c,33h
    mvi l,4ch
    sub l
    jc blkfnd
    inr h
    ;in block 3, offset of start of
    ; game data=1*256, and there are
    ; 7eh*256 bytes of game data
    mvi c,1
    mvi l,7eh
    sub l
    jc blkfnd
    inr h
    ;in block 4, offset of start of
    ; game data=01h*256 (already set)
    ; and upper-64k game data runs out
    ; in this block
    sub l
    jmp blkfnd
gtlw64:
    ;read from lower 64k of game rom
    mvi h,0 ;rexblock[h] REX block num
    mov a,d
    ;in block 0, offset of start of
    ; game data=2fh*256, and there are
    ; 50h*256 bytes of game data
    mvi c,2fh
    mvi l,50h
    sub l
    jc blkfnd
    inr h
    ;in block 1, offset of start of
    ; game data=01h*256, and there
    ; are 7eh*256 bytes of game data
    mvi c,01h
    mvi l,7eh
    sub l
    jc blkfnd
    inr h
    ;in block 2, offset of start of
    ; game data=01h*256 (already set)
    ; and lower-64k game data runs out
    ; in this block
    sub l
blkfnd:
    add l
    add c
    mov d,a ;de is src adx for REX blk
    mov a,h
    lxi h,rexblock
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    mov a,m ;src REX blk
    pop h   ;dst adx
    ;b still has #bytes to get
    jmp zgtabl

;puts array of bytes to z-machine mem
;put is always to lower 64k of zmem
;inpars:
; a is the number of bytes ([1,4])
; de pts to source array in local mem
; 0hl adx of z-machine mem destination        
;outpar: none
;MAY DESTROY ALL REGS
zputa:
    mov b,a ;#bytes saved in b
    push d
    xchg
    ;make sure address in dynamic zmem
    lhld mdynsize
    dad d
    cc error
    ;de is an offset into dynamic zmem
    ; which is at dynbase in ram
    ;make hl the ram address
    lhld dynbase
    dad d ;hl dst adx
    pop d ;de src adx
    ;do the copy
    ldax d
    mov m,a
    dcr b
    rz
    inx d
    inx h
    ldax d
    mov m,a
    dcr b
    rz
    inx d
    inx h
    ldax d
    mov m,a
    dcr b
    rz
    inx d
    inx h
    ldax d
    mov m,a
    ret
    

;;;gets a 32-bit integer from RAM
;;;inpar: address in hl
;;;outpars: the val in bcde
;gets a 24-bit integer from RAM
;inpar: address in hl
;outpars: the val in cde
;PRESERVES hl
getl:
    push h
    mov e,m
    inx h
    mov d,m
    inx h
    mov c,m
;;    inx h
;;    mov b,m
    pop h
    ret

;;;puts a 32-bit integer to RAM at hl
;;;inpars: address in hl, val in bcde
;;;outpar: none
;puts a 24-bit integer to RAM at hl
;inpars: address in hl, val in cde
;outpar: none
;PRESERVES hl
putl:
    push h
    mov m,e
    inx h
    mov m,d
    inx h
    mov m,c
;;    inx h
;;    mov m,b
    pop h
    ret

;init code to be called when game
; starting, to set up reading from
; game image - need to call this
; before using zgeta, zputa
;DESTROYS ALL REGS
setupimage:
    ;the block currently selected is
    ; first (contains interpreter)
    call getrex
    sta rexblock
    ;get game name
    lxi h,gamename
    lxi d,myname
    lxi b,5
    call memcopy
    ;this is the part that must execute
    ; from ram:
    call siram
    ;make sure we got the expected
    ; number of blocks
    lda numblocks
    mov c,a
    mvi b,5
    lxi h,rexblock
cbklp:
    mov a,m
    ana a
    jz cbklpe
    inx h
    dcr c
    dcr b
    jnz cbklp
cbklpe:
    mov a,c
    ana a
    cnz error
    ;put interpreter block in unused
    ; rexblock entries so a bad game can't
    ; access outside of its own image
    lda rexblock
    mov c,a
stpzbn:
    dcr b
    jm stpram
    mov a,m
    ana a
    cnz error
    mov m,c
    inx h
    jmp stpzbn
stpram:
    ;copy game image dynamic mem to ram
    lhld dynsize
    ;we copy 9 bytes past the end
    ; of game img dyn mem so that
    ; we can read up to 10 bytes
    ; from ram copy of dyn mem if
    ; starting in dyn mem adxs
    lxi d,9
    dad d
    mov b,h
    mov c,l
    lhld dynbase
    xchg        ; de=dst
    lxi h,roimg ; hl=src
    jmp memcopy
    
    
; ******** RAM INTERRUPT CODE ******* ;
; This is the ram portion of the      ;
; interrupt code, which gets copied   ;
; on startup into ram.                ;

rmicd:
rramint:
;;ramint: ;interrupts are disabled
    out 0e8h ;switch to STDROM
    ret ;jmp to STDROM int. vector
rramint2:
;;ramint2: ;STDROM int vec returns here
    di
    lda 0ff45h
    ori 01h
    sta 0ff45h
    out 0e8h ;switch to OPTROM
    pop psw
    pop h
    ei
    ret
rexit:
;;exit: ;does reset in STDROM
    lxi sp,tmpstackend ;temp stack for call  ;; TODO is this still needed?
rexit2:
    di
    lxi h,0 ;intvec = reset
    jmp int2
rmicde:


; ********* RAM FUNCTIONS *********** ;
; (Functions that need to be in ram.) ;
; The code from ramfuncsromcopystart  ;
; to ramfuncsromcopyend will be       ;
; copied to ram on startup.           ;

ramfuncsromcopystart:

;selects MAIN ROM
xxxmanrom:
    di
    lda 0ff45h
    ani 0feh
    sta 0ff45h
    out 0e8h
    ei
    ret

;selects OPT ROM
xxxoptrom:
    di
    lda 0ff45h
    ori 001h
    sta 0ff45h
    out 0e8h
    ei
    ret

;calls lnkfil
xxxmylnkfil:
    push b
    push d
    push h
    call manrom
    call lnkfil
    call optrom
    pop h
    pop d
    pop b
    ret

;calls srccom
xxxmysrccom:
    push b
    call manrom
    call srccom
    push psw
    call optrom
    pop psw
    pop b
    ret
    
;calls kilcom
xxxmykilcom:
    push b
    push d
    push h
    call manrom
    call kilcom
    call optrom
    pop h
    pop d
    pop b
    ret
    
;calls makhol
xxxmymakhol:
    push b
    push d
    push h
    call manrom
    call makhol
    call optrom
    pop h
    pop d
    pop b
    ret

;initialization for screen output
;inpar: none
;outpar: none
;MODIFIES ALL REGS
;on exit, OPT ROM is selected
xxxscinit:
    call manrom
    ;do init
    call 5a79h;CLRFLK clear f.key defs
    call 428ah;ERAFNK hide f.key labels
    mvi a,0
    sta 0faadh ;label line disabled
    call 422dh;HOME   cursor top left
    call 4231h;CLS    clear screen
    call 4244h;UNLOCK allow scrolling
    call optrom
    ret
    
;saves regs over a call to LCD, which
; displays a character to the screen
;inpar: a is the character to display
;outpar: none
;PRESERVES a
;on exit, OPT ROM is selected
xxxsafelcd:
    push b
    push d
    push h
    push psw
    mov b,a
    call manrom
    ;call LCD
    mov a,b
    call lcd
    ;back to OPT ROM
    call optrom
    pop psw
    pop h
    pop d
    pop b
    ret

;saves regs over a call to CHGET,
; which gets a character from keyboard
;also ignores "special" characters
;inpar: none
;outpar: char in a
;on exit, OPT ROM is selected
xxxsafechget:
    push b
    push d
    push h
    call manrom
xxxcgetlp:
    call 12cbh ;CHGET
    jc cgetlp  ;ignore "special" chars
    mov b,a
    ;back to OPT ROM
    call optrom
    mov a,b
    pop h
    pop d
    pop b
    ret

;copies bytes from given REX block to
; array in ram
;inpars:
;   b is #bytes ([1,4])
;   a is REX block number ([6,31])
;   de is src adx in block ([0,32767])
;   hl is dst adx in ram
;MAY DESTROY ALL REGS
xxxzgtabl:
    di
    call selrex
    ei
xxxzgtlp:
    ldax d
    mov m,a
    inx d
    inx h
    dcr b
    jnz zgtlp
    lda rexblock ;first block
    di
    call selrex
    ei
    ret

    
;this is the portion of setupimage that
; must be in ram
;DESTROYS ALL REGS
xxxsiram:
    ;check all the general use REX blks
    ; (6 to 31) for more game image
    ; blocks
    mvi c,5
xxxstplp:
    inr c
    mov a,c
    cpi 32
    jnc stplpe
    ;check if this REX block is valid
    call bvalid ;preserves a
    jz stplp
    ;select block, check for game name
    di
    call selrex
    ei
    lxi h,gamename
    lxi d,myname
    mvi b,5
xxxcnamlp:
    ldax d
    cmp m
    jnz stplp
    inx d
    inx h
    dcr b
    jnz cnamlp
    ;name matched
    ;next char is block number
    mov a,m
    sui '0'
    cpi 5
    jnc stplp ;can only be 0,1,2,3,4
    lxi h,rexblock
    ana a
    jz stpbk0
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    mov a,m     ;rexblock[blocknum]
    ana a
    cnz stperr  ;duplicate block
    mov m,c
    jmp stplp
xxxstpbk0:
    lda rexblock
    cmp c
    cnz stperr  ;duplicate block 0
    jmp stplp
xxxstperr:
    lda rexblock
    di
    call selrex ; back to interp. blk
    ei
    jmp error
xxxstplpe:
    lda rexblock
    di
    call selrex ; back to interp. blk
    ei
    ret

    
;cleanup code related to game image,
; to be called when game ending
xxxclnimg:
    ret

;determines what REX block is currently
; selected
;OPT ROM must be selected before this
; is called
;leaves OPT ROM selected
;leaves w/ REX state mach. in st 111
;inpar: none
;outpar: REX block number in b
xxxgetrex:
    di
    ;
    ;send key to REX to put REX state
    ; machine into state 000
    ;
    call rexkey
    ;
    ;now in state 000
    ;
    ;read the status register
    push h
    lxi h,0003h
    mov a,m ;read status
    ani 3fh ;sector number
    mov b,a
    mov a,m ;back to state 000
    ; go to state 111
	lxi	h,0000h
	mov	a,m
	mov	a,m
	mov	a,m ;3 reads from 0
            ; makes it go to state 111
    mov a,b
    pop h
    ei
    sta currex
    ret

;selects REX block (must be [6,31])
;OPT ROM must be selected before this
; is called
;interrupts must be disabled before
; this is called
;leaves OPT ROM selected
;expects REX state mach. in st 111
;leaves w/ REX state mach. in st 111
;inpar: a is the REX block number
;outpar: none
xxxselrex:
    push h
    lxi h,currex
    cmp m
    jz selrxr;correct block selected
    sta currex
    ;
    ;send key to REX to put REX state
    ; machine into state 000
    ;
    call rexkey
    ;
    ;now in state 000
    ;
    ;send the SET BLOCK command
    ;
    lda currex
	lxi	h,0001h
	mov	l,m
	mov	l,a		
	mov	a,m
    ;
    ; now in state 000
    ; go to state 111
	lxi	h,0000h
	mov	a,m
	mov	a,m
	mov	a,m ;3 reads from 0
            ; makes it go to state 111
xxxselrxr:
    pop h
	ret

;helper for getrex, selrex
;sends REX unlock key, leaving REX
; state machine in state 000
xxxrexkey:
    ;send key to REX to put REX state
    ; machine into state 000
    ;
	push	h
	push	psw
	lxi	h,0000h
	mov	a,m
	mov	a,m
	mov	a,m ;3 reads from 0
            ; makes it go to state 111
	mvi	l,184d		; key = B8
	mov	l,m		
	mvi	l,242d		; key = F2
	mov	l,m		
	mvi	l,52d		; key = 34
	mov	l,m		
	mvi	l,176d		; key = B0
	mov	l,m		
	mvi	l,49d		; key = 31
	mov	l,m		
	mvi	l,191d		; key = BF
	mov	l,m
	pop	psw
	pop	h
    ret
    
;determine if a given REX block is
; valid
;OPT ROM must be selected before this
; is called
;leaves OPT ROM selected, with REX
; block rexblock[0] selected as OPT ROM
;inpar: a is REX block number ([6,31])
;outpars: z=0 iff valid,
;      if valid, hl->directory entry
;PRESERVES a
xxxbvalid:
    push b
    mov c,a
    ori 0c0h ;valid entries have these
    mov b,a  ;search for this
    ;select REX block 0
    mvi a,0
    di
    call selrex
    ;search directory for the blocknum
    lxi h,namdir
xxxdrsrch:
    mov a,m
    inr a
    jz bnf ;reached directory end
    dcr a  ;1st byte of dir entry
    cmp b
    jz bfnd
    ;advance to next dir entry
    mvi a,16
    add l
    mov l,a
    mvi a,0
    adc h
    mov h,a
    jmp drsrch
xxxbfnd:
    ;block found
    lda rexblock
    call selrex
    ei
    mvi a,1
    ana a
    mov a,c
    pop b
    ret
xxxbnf:
    ;block not found
    lda rexblock
    call selrex
    ei
    xra a
    mov a,c
    pop b
    ret

ramfuncsromcopyend:



; ******* FIXED RAM LOCATIONS ******* ;
; We use inpbuf and altlcd for:       ;
; * ram functions offset table        ;
; * ram portion of the interrupt code ;
; * variables                         ;

; ************ INPBUF *************** ;
; **** RAM FUNCTIONS OFFSET TABLE *** ;
; The following table contains all    ;
; the entry points to code that we    ;
; will put in ram, as offsets from    ;
; start of the ram functions.         ;
; Each entry gets 3 bytes; enough to  ;
; store  'jmp <address>'.             ;

offsettablestart: ;rom start adx of tbl
rfjmpt equ inpbuf
manrom equ rfjmpt
    dw (xxxmanrom - ramfuncsromcopystart)
optrom equ (manrom + 3)
    dw (xxxoptrom - ramfuncsromcopystart)
mylnkfil equ (optrom + 3)
    dw (xxxmylnkfil - ramfuncsromcopystart)
mysrccom equ (mylnkfil + 3)
    dw (xxxmysrccom - ramfuncsromcopystart)
mykilcom equ (mysrccom + 3)
    dw (xxxmykilcom - ramfuncsromcopystart)
mymakhol equ (mykilcom + 3)
    dw (xxxmymakhol - ramfuncsromcopystart)
scinit equ (mymakhol + 3)
    dw (xxxscinit - ramfuncsromcopystart)
safelcd equ (scinit + 3)
    dw (xxxsafelcd - ramfuncsromcopystart)
safechget equ (safelcd + 3)
    dw (xxxsafechget - ramfuncsromcopystart)
cgetlp equ (safechget + 3)
    dw (xxxcgetlp - ramfuncsromcopystart)
zgtabl equ (cgetlp + 3)
    dw (xxxzgtabl - ramfuncsromcopystart)
zgtlp equ (zgtabl + 3)
    dw (xxxzgtlp - ramfuncsromcopystart)
siram equ (zgtlp + 3)
    dw (xxxsiram - ramfuncsromcopystart)
stplp equ (siram + 3)
    dw (xxxstplp - ramfuncsromcopystart)
cnamlp equ (stplp + 3)
    dw (xxxcnamlp - ramfuncsromcopystart)
stpbk0 equ (cnamlp + 3)
    dw (xxxstpbk0 - ramfuncsromcopystart)
stperr equ (stpbk0 + 3)
    dw (xxxstperr - ramfuncsromcopystart)
stplpe equ (stperr + 3)
    dw (xxxstplpe - ramfuncsromcopystart)
clnimg equ (stplpe + 3)
    dw (xxxclnimg - ramfuncsromcopystart)
getrex equ (clnimg + 3)
    dw (xxxgetrex - ramfuncsromcopystart)
selrex equ (getrex + 3)
    dw (xxxselrex - ramfuncsromcopystart)
selrxr equ (selrex + 3)
    dw (xxxselrxr - ramfuncsromcopystart)
rexkey equ (selrxr + 3)
    dw (xxxrexkey - ramfuncsromcopystart)
bvalid equ (rexkey + 3)
    dw (xxxbvalid - ramfuncsromcopystart)
drsrch equ (bvalid + 3)
    dw (xxxdrsrch - ramfuncsromcopystart)
bfnd equ (drsrch + 3)
    dw (xxxbfnd - ramfuncsromcopystart)
bnf equ (bfnd + 3)
    dw (xxxbnf - ramfuncsromcopystart)
; end address of jump table in rom:
offsettableend:

; ************* INPBUF ************** ;
; ************ VARIABLES ************ ;
svwrite equ (bnf + 3)
rswrite equ svwrite ;share same space
svwnext equ (svwrite + 3)
svwritefilestart equ (svwnext + 2)
svrdnext equ (svwritefilestart + 2)
rgnext equ (svrdnext + 2)
pframe equ (rgnext + 2)
wordsused equ (pframe + 2)
numlocals equ (wordsused + 2)
temp16 equ (numlocals + 1)
temp8 equ (temp16 + 2)
formlengthloc equ (temp8 + 1)
chunklengthloc equ (formlengthloc + 2)
zerocount equ (chunklengthloc + 2)
dirslot equ (zerocount + 1)
numevalwords equ (dirslot + 2)
callerszlp equ (numevalwords + 2)
formlength equ (callerszlp + 2)
formstartloc equ (formlength + 2)
chunklength equ (formstartloc + 2)
chunkstartloc equ (chunklength + 2)
restorezpc equ (chunkstartloc + 2)
gotcmem equ (restorezpc + 4)
gotstks equ (gotcmem + 1)
rsrdnext equ (gotstks + 1)
rsrdend equ (rsrdnext + 2)
rswrnext equ (rsrdend + 2)
rswrend equ (rswrnext + 2)
restorezsp equ (rswrend + 2)
; end address of the portion of inpbuf
;  that we've used:
inpbufusedend equ (restorezsp + 2)
; !!! TODO verify
;   inpbufusedend <= inpbufend

; ************** ALTLCD ************* ;
; (See http://bitchin100.com/wiki/index.php?title=Model_100_System_Map_Part_7_(F5F4-FFEC)) ;
; The ram portion of interrupt code,  ;
; as well as most variables, go in    ;
; "alternate LCD character buffer".   ;

;ram portion of interrupt code
ramint equ altlcd
ramint2 equ (ramint + rramint2 - rramint)
exit equ (ramint + rexit - rramint)
exit2 equ (ramint + rexit2 - rramint)

; ************* ALTLCD ************** ;
; ************ VARIABLES ************ ;
tmpstack equ (ramint + rmicde - rmicd)
tmpstackend equ (tmpstack + 7)
dynbase equ (tmpstackend + 1)
dynsize equ (dynbase + 2)
mdynsize equ (dynsize + 2) ; -dynsize
dyntop equ (mdynsize + 2)
zstk equ (dyntop + 2)
ramfuncs equ (zstk + 2)
;START of memory to be zeroed on
; (re)start
zeroinitmem equ (ramfuncs + 2)
; **** z-machine variables **** ;
zsp equ (ramfuncs + 2) ;stack pointer
zlp equ (zsp + 2) ;locals pointer
zpc equ (zlp + 2) ;program counter
zstatbase equ (zpc + 4)
zdictbase equ (zstatbase + 2)
zdseps equ (zdictbase + 2)
zdsepsend equ (zdseps + 9)
zdentrylength equ zdsepsend
zdnumentries equ (zdentrylength + 1)
zobjectsbase equ (zdnumentries + 2)
zobjectstable equ (zobjectsbase + 2)
zglobalsbase equ (zobjectstable + 2)
zabbrevsbase equ (zglobalsbase + 2)
; *** interpreter variables *** ;
temp32 equ (zabbrevsbase + 2)
tempcallinfom equ (temp32 + 4)
instbuf equ (tempcallinfom + 2)
instbufptr equ (instbuf + 10)
dictword equ (instbufptr + 2)
wdcharsleft equ (dictword + 6)
dictwdcharsleft equ (wdcharsleft + 1)
packedword equ (dictwdcharsleft + 1)
randtemp equ (packedword + 4)
predseed equ (randtemp + 4)
prevrand equ (predseed + 2)
stradx equ (prevrand + 4)
packed3 equ (stradx + 3)
packed3count equ (packed3 + 2)
inabr equ (packed3count + 1)
linecount equ (inabr + 1)
linebuf equ (linecount + 1)
lbcount equ (linebuf + 41)
lbspacecount equ (lbcount + 1)
lbnextptr equ (lbspacecount + 1)
txtbuf equ (lbnextptr + 2)
parsebufpos equ (txtbuf + 2)
pbentriesleft equ (parsebufpos + 2)
opcode equ (pbentriesleft + 1)
opnum equ (opcode + 1)
opcount equ (opnum + 1)
count equ (opcount + 1)
operands equ (count + 1)
op1 equ operands
op2 equ (op1 + 2)
op3 equ (op2 + 2)
op4 equ (op3 + 2)
rexblock equ (op4 + 2)
currex equ (rexblock + 5)
myname equ (currex + 1)
;END of memory to be zeroed on
; (re)start
zeroinitmemend equ (myname + 5)
; end address of the portion of altlcd
;  that we've used:
altlcdusedend equ (restorezsp + 2)
; !!! TODO verify
;   altlcdusedend <= altlcdend


    dw inpbufusedend
    dw inpbufend
    dw altlcdusedend
    dw altlcdend

; ***** END OF INTERPRETER CODE ***** ;

; ***** GAME IMAGE STARTS HERE: ***** ;
roimg equ 12032
;(note that .aseg ends at 12031)

end

