#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

/* packaging program for z100, an interpreter of version 3 z-machine game files */
/* for the TRS Model 100 portable computer */
/*
; Copyright (C) 2015 Clinton Reddekop.
; You can redistribute and/or modify
; this program under the terms of the
; GNU General Public License as
; published by the Free Software
; Foundation, either version 3 of the
; License, or (at your option) any
; later version.
; This program is distributed in the
; hope that it will be useful, but
; WITHOUT ANY WARRANTY; without even
; the implied warranty of
; MERCHANTABILITY or FITNESS FOR A
; PARTICULAR PURPOSE. See the GNU
; General Public License (available at
; <http://www.gnu.org/licenses/gpl-3.0.html>)
; for details.
;*/

#include "interp.h"

void mfputc(char c, FILE *ofp);
char mfgetc(FILE *ifp);
char *filetoarray(char *filename, int *pfilesize);
int arraytofile(char *filename, char *pdata, int numbytes);

int main(int argc, char **argv) {
    
    char gamename[6];
    int i;
    char c;
    int block;
    int gameoffset;
    char filename[11];
    char blocknum[2];
    char *gnp;
    char *pinterp;
    char *gamearray;
    char *pgame;
    static char outarray[32*1024];
    char *pout;
    char *pgameend;
    int gamelength;
    int hdrgamelength;
    int dynsize;
    
    
    if (argc != 2) {
        fprintf(stderr, "usage: z100pkg <datfilename>\n"
            " where <datfilename> is the name of a version 3 z-machine game file.\n");
        return 1;
    }
    gamearray = filetoarray(argv[1], &gamelength);
    if (!gamearray) {
        fprintf(stderr, "\nERROR reading game file '%s'\n", argv[1]);
        return 2;
    }
    if (gamelength < 64) {
        free(gamearray);
        fprintf(stderr, "\nERROR game file '%s' too short\n", argv[1]);
        return 3;
    }
    if (gamearray[0] != 3) {
        free(gamearray);
        fprintf(stderr, "\nERROR wrong version '%s' - only version 3 is allowed\n",
            argv[1]);
        return 4;
    }
    
    if (gamelength > 128*1024) {
        /* right now gamelength is the size of the game file read, but the actual game
           image in that file may be smaller - read actual length from header if there */
        hdrgamelength = ((int)gamearray[0x1a] & 0x0ff) << 8 | ((int)gamearray[0x1b] & 0x0ff);
        hdrgamelength *= 2;
        if (hdrgamelength != 0) {
                gamelength = 128*1024;
        } else {
            fprintf(stderr, "\nERROR game file '%s' too long\n", argv[1]);
            free(gamearray);
            return 6;
        }
    }
    if (gamelength < 64) {
        fprintf(stderr, "\nERROR game file '%s' too short\n", argv[1]);
        free(gamearray);
        return 7;
    }
    
    /* check that the size of the game's dynamic memory is <= 20480 - this is the max
       that the interpreter can handle */
    dynsize = ((int)gamearray[0x0e] << 8 & 0x0ff00) | ((int)gamearray[0x0f] & 0x0ff);
    if (dynsize > 20480) {
        fprintf(stderr, "\nERROR too much dynamic memory required\n");
        return 8;
    }
    
    /* number of game blocks we expect to write goes in interp[0x49] */
    if (gamelength <= 20480) {
        interp[0x49] = 1;
    } else if (gamelength <= 20480+32256) {
        interp[0x49] = 2;
    } else if (gamelength <= 20480+32256+32256) {
        interp[0x49] = 3;
    } else if (gamelength <= 20480+32256+32256+32256) {
        interp[0x49] = 4;
    } else {
        interp[0x49] = 5;
    }
    
    /* get game file name up to 5 chars */
    for (i = 0; i < 5; ++i) {
        c = argv[1][i];
        if (!c || c=='.') {
            break;
        }
        gamename[i] = c;
    }
    gamename[i] = 0;
    
    /* game name goes in 0x42-0x46 of interpreter - pad w/ ' ' as needed */
    gnp = gamename;
    for (i = 0x42; i < 0x47; ++i) {
        c = *gnp;
        if (!c) {
            break;
        }
        ++gnp;
        interp[i] = c;
    }
    for ( ; i < 0x47; ++i) {
        interp[i] = ' ';
    }
    
    
    /* zeroth block, then first, second, third, fourth blocks as needed */
    pgame = gamearray;
    pgameend = pgame + gamelength;
    blocknum[1] = 0;
    for (block = 0; block < 5; block++) {
        blocknum[0] = block + '0';
        interp[0x47] = blocknum[0]; /* block num after game name */
        /* zeroth block starts with the entire interpreter program */
        /* every block gets start of interpreter program
           this ensures that all blocks get interrupt code and game name, as well
           as padding up to the offset where the next portion of the game image
           should go */
        if (block == 0) {
            gameoffset = 12032; /* copy up to this byte of interpreter */
        } else {
            /* first, second or third block */
            gameoffset = 256; /* copy up to this byte of interpreter */
        }
        pout = outarray;
        pinterp = interp;
        for (i = 0; i < gameoffset; ++i) {
            *pout = *pinterp;
            ++pout;
            ++pinterp;
        }
        
        /* fill with game image bytes up to 32640 */
        for ( ; i < 32640; ++i) {
            if (pgame < pgameend) {
                *pout = *pgame;
            } else {
                *pout = 0;
            }
            ++pout;
            ++pgame;
        }
        /* pad with zeros */
        for ( ; i < sizeof(outarray); ++i) {
            *pout = 0;
            ++pout;
        }
        /* write output file */
        strcpy(filename, gamename);
        strcat(filename, blocknum);
        strcat(filename, ".bx");
        if (!arraytofile(filename, outarray, sizeof(outarray)))
        {
            fprintf(stderr, "\nERROR writing output file '%s'\n", filename);
            free(gamearray);
            return 9;
        }
        
        /* we want a 128-byte overlap between consecutive game image blocks */
        pgame -= 128;
        
        /* done yet? */
        if (pgame >= pgameend) {
            printf("\nwrote %d files\n", block+1);
            free(gamearray);
            return 0;
        }
    }
    
    /* shouldn't get here :( */
    fprintf(stderr, "\nERROR game file '%s' too big for 5 blocks (?)\n", argv[1]);
    free(gamearray);
    return 10;
}

void mfputc(char c, FILE *ofp) {
    if (ferror(ofp)) {
        return;
    }
    fputc(c, ofp);
}

char mfgetc(FILE *ifp) {
    char c;
    if (feof(ifp) || ferror(ifp)) {
        return 0;
    }
    c = fgetc(ifp);
    if (feof(ifp) || ferror(ifp)) {
        return 0;
    }
    return c;
}

/* opens file as binary, writes numbytes to it from *pdata, closes file */
/* returns nonzero on success, zero on failure */
int arraytofile(char *filename, char *pdata, int numbytes) {
    FILE *ofp;
    
    ofp = fopen(filename, "wb");
    if (!ofp) {
        return 0;
    }
    
    for ( ; numbytes > 0 && !ferror(ofp); --numbytes) {
        fputc(*pdata, ofp);
        ++pdata;
    }
    if (ferror(ofp)) {
        fclose(ofp);
        return 0;
    }
    
    fclose(ofp);
    return 1;
}

/* opens file as binary, reads into malloc'd array, closes file */
/* if any error occurs returns null, else returns address of array */
/* must have 1 <= file size <= INT_MAX */
char *filetoarray(char *filename, int *pfilesize) {
    FILE *ifp;
    long int lfilesize;
    int filesize;
    int i;
    char *parray;
    
    *pfilesize = 0;
    
    ifp = fopen(filename, "rb");
    if (!ifp) {
        return 0;
    }
    if (fseek(ifp, 0, SEEK_END)) {
        fclose(ifp);
        return 0;
    }
    if (ferror(ifp)) {
        fclose(ifp);
        return 0;
    }
    lfilesize = ftell(ifp);
    if (lfilesize < 1 || lfilesize > INT_MAX) {
        fclose(ifp);
        return 0;
    }
    filesize = (int)lfilesize;
    rewind(ifp);
    if (ferror(ifp)) {
        fclose(ifp);
        return 0;
    }
    parray = malloc(filesize);
    if (!parray) {
        fclose(ifp);
        return 0;
    }
    
    /* read file into array */
    for(i = 0; i < filesize; ++i) {
        parray[i] = mfgetc(ifp);
    }
    if (ferror(ifp) || feof(ifp)) {
        free(parray);
        fclose(ifp);
        return 0;
    }
    fgetc(ifp);
    if (!feof(ifp)) {
        free(parray);
        fclose(ifp);
        return 0;
    }
    
    fclose(ifp);
    *pfilesize = filesize;
    return parray;
}
